#' @title
#' Getting poset elements.
#'
#' @description
#' Gets the elements of the ground set \eqn{V} of the input poset \eqn{(V,\leq)}.
#'
#' @param poset An object of S4 class `POSet`
#' Argument `poset` must be created by using any function contained in the package aimed at building object of S4 class `POSet`
#' (e.g. [POSet()], [LinearPOSet()], [ProductPOSet()], ...) .
#'
#' @return
#' A vector of labels (the names of the elements of the ground set \eqn{V}).
#'
#' @examples
#' elems <- c("a", "b", "c", "d")
#'
#' dom <- matrix(c(
#'   "a", "b",
#'   "c", "b",
#'   "b", "d"
#' ), ncol = 2, byrow = TRUE)
#'
#' pos <- POSet(elements = elems, dom = dom)
#'
#' gset <- POSetElements(pos)
#'
#' @name POSetElements
#' @export POSetElements
POSetElements <- function(poset) {
  if (!methods::is(poset, "POSet")) {
    stop("Argument poset must be of class POSet")
  }
  tryCatch({
    result <- .Call("_Elements", poset@ptr)
    return (result)
  }, error = function(err) {
    err_split <- strsplit(err[[1]], split = ":")
    stop(err_split[[1]][length(err_split[[1]])])
  }) # END tryCatch
}
