#' Plot Estimated Return Levels for NS-FFA
#'
#' Generates a plot with effective return periods on the x-axis and effective return 
#' levels (annual maxima magnitudes) on the y-axis. Each slice is displayed in a 
#' distinct color. Confidence bounds are shown as semi-transparent  ribbons, and the 
#' point estimates  are overlaid as solid lines. Return periods have a logarithmic scale.
#'
#' @param results A fitted flood frequency model generated by [fit_lmoments()], 
#' [fit_mle()] or [fit_gmle()] OR a fitted model with confidence intervals generated
#' by [uncertainty_bootstrap()], [uncertainty_rfpl()], or [uncertainty_rfgpl()].
#'
#' @param slices Default time slices for plotting the return levels if confidence
#' intervals are not provided.
#' 
#' @inheritParams param-periods
#'
#' @param ... Optional named arguments: 'title', 'xlabel', and 'ylabel'.
#'
#' @return `ggplot`; a plot with one line and ribbon per slice.
#'
#' @examples
#'
#' # Fit a nonstationary model  
#' data <- rnorm(n = 100, mean = 100, sd = 10)
#' years <- seq(from = 1901, to = 2000)
#' ns_structure <- list(location = TRUE, scale = FALSE)
#'
#' results <- fit_mle(
#' 	   data, 
#' 	   "GEV", 
#' 	   ns_years = years, 
#' 	   ns_structure = ns_structure
#' )
#'
#' # Generate the plot
#' plot_nsffa_estimates(results)
#'
#' @import ggplot2
#' @export

plot_nsffa_estimates <- function(
	results,
	slices = c(1900, 1940, 1980, 2020),
	periods = c(2, 5, 10, 20, 50, 100),
	...
) {

	# Override the periods argument if contained in 'results'
	if ("ci_list" %in% names(results)) {
		periods = results$ci_list[[1]]$periods
	}

    # Capture optional arguments
    args <- list(...)
    title  <- args$title  %||% "NS-FFA Results"
    xlabel <- args$xlabel %||% "Effective Return Period (Years)"
    ylabel <- args$ylabel %||% expression("Streamflow (" * m^3/s * ")")
  
    # Define color palette
    palette <- c("#541352", "#3A5E8C", "#2F9AA0", "#10A53D", "#FFCF20")
  
    # Build combined data frame with an identifier for each slice
	slices <- if ("ci_list" %in% names(results)) {
		as.numeric(names(results$ci_list))
	} else {
		slices 
	}

    df_list <- lapply(seq_along(slices), function(i) {

		# Compute the quantile estimates
        slice <- slices[i]
		returns = 1 - (1 / periods)
		structure = list(location = FALSE, scale = FALSE)

		estimates = utils_quantiles(
			returns,
			results$distribution,
			results$params,
			slice,
			results$ns_structure
		)

		# Store quantile estimates in a dataframe  
        df <- data.frame(
            periods   = periods,
            estimates = estimates,
            slice     = as.character(slice)
        )

		# Add the confidence intervals if given
		if ("ci_list" %in% names(results)) {
            df$lower = results$ci_list[[i]]$lower
            df$upper = results$ci_list[[i]]$upper
		}

		# Return the dataframe
		df

    })
  
    df_all <- do.call(rbind, df_list)
  
    # Create a plot of the return level estimates
    p <- ggplot(df_all, aes(x = .data$periods, y = .data$estimates)) +
        geom_line(aes(color = .data$slice), linewidth = 1) +
        geom_point(aes(color = .data$slice), size = 2.25) 

	# Add the confidence intervals if given
	if ("ci_list" %in% names(results)) {
		p <- p + geom_ribbon(
			aes(ymin = .data$lower, ymax = .data$upper, fill = .data$slice),
			alpha = 0.2
		)
	}

	# Add the color scales and labels 
	p <- p + scale_color_manual(values = palette[seq_along(slices)]) +
        scale_fill_manual(values = palette[seq_along(slices)]) +
        scale_x_log10(breaks = unique(df_all$periods)) +
        labs(title = title, x = xlabel, y = ylabel, color = "Return Levels") +
  		guides(fill = "none")
  
    # Add theme and return
    add_theme(p)
}

