#' Coordinates for data from brain atlases
#'
#' Datasets containing spatial coordinates for: the original AAL atlases, the
#' newer AAL2 atlases, Freesurfer atlases, Brainsuite, Craddock200,
#' Dosenbach160, Harvard-Oxford, and LONI probabilistic brain atlas. In addition
#' to coordinates, there are indices for the major lobes and hemispheres of the
#' brain, the \emph{class} variable (for Destrieux atlases), functional networks
#' (for Dosenbach, Power, and Gordon atlases; plus the Yeo network labels for
#' the Brainnetome atlas).
#'
#' @format A data frame with 90 or 116 (for the original AAL atlases), 94 or 120
#' (for the newer AAL2 atlases), 148 or 162 (for Destrieux), 68 or 82 (for DK),
#' 62 or 76 (for DKT), 74 (Brainsuite), 200 (Craddock), 160 (Dosenbach), 112
#' (Harvard-Oxford), 40 (LONI), 246 (Brainnetome), 360 (HCP), 264 (Power), or
#' 333 (Gordon) observations on (some of) the following 19 variables:
#' \describe{
#'   \item{\code{name}}{a character vector of region names}
#'   \item{\code{x.mni}}{a numeric vector of x-coordinates (in MNI space)}
#'   \item{\code{y.mni}}{a numeric vector of y-coordinates (in MNI space)}
#'   \item{\code{z.mni}}{a numeric vector of z-coordinates (in MNI space)}
#'   \item{\code{lobe}}{a factor with some of levels \code{Frontal}
#'     \code{Parietal} \code{Temporal} \code{Occipital} \code{Insula}
#'     \code{Limbic} \code{Cingulate} \code{SCGM} \code{Cerebellum} (for
#'     \code{aal116} and \code{aal2.120}) and \code{Brainstem} (for
#'     \code{craddock200})}
#'   \item{\code{hemi}}{a factor with levels \code{L} \code{R} and \code{B} (for
#'     \code{dosenbach160})}
#'   \item{\code{index}}{a numeric vector}
#'   \item{\code{name.full}}{a character vector of full region names, for the DK
#'     and DKT atlases}
#'   \item{\code{class}}{a factor with levels \code{G} \code{G_and_S} \code{S},
#'     for the Destrieux atlases}
#'   \item{\code{network}}{(dosenbach160) a factor with levels \code{default}
#'     \code{fronto-parietal} \code{cingulo-opercular} \code{sensorimotor}
#'     \code{cerebellum} \code{occipital}}
#'   \item{\code{gyrus}}{(brainnetome) Abbreviated names of gyri/regions
#'     (including subcortical), with 24 unique values}
#'   \item{\code{gyrus.full}}{(brainnetome) Full names of \code{gyrus}}
#'   \item{\code{subregion}}{(brainnetome) Abbreviated names of subregions
#'     (including subdivisions of subcortical gray matter)}
#'   \item{\code{subregion.full}}{(brainnetome) Full names of \code{subregion}}
#'   \item{\code{Yeo_7network}}{(brainnetome) Factor with 8 levels consisting of
#'     SCGM plus the 7 networks from Yeo et al.}
#'   \item{\code{Yeo_17network}}{(brainnetome) Factor with 18 levels consisting
#'     of SCGM plus the 17 networks from Yeo et al.}
#'   \item{\code{area}}{(HCP) a factor with 23 cortical areas}
#'   \item{\code{Anatomy}}{(power264) Full region/gyrus names for the Power
#'     atlas; contains 53 unique regions}
#'   \item{\code{Brodmann}}{(power264) Integer values for Brodmann areas}
#' }
#'
#' @name Brain Atlases
#' @aliases aal116
#' @rdname brain_atlases
#' @references Tzourio-Mazoyer, N. and Landeau, B. and Papathanassiou, D. and
#'   Crivello, F. and Etard, O. and Delcroix, N. and Mazoyer, B. and Joliot, M.
#'   (2002) Automated anatomical labeling of activations in SPM using a
#'   macroscopic anatomical parcellation of the MNI MRI single-subject brain.
#'   \emph{NeuroImage}, \bold{15(1)}, 273--289.
#'   \doi{10.1006/nimg.2001.0978}
#' @references Rolls, E.T. and Joliot, M. and Tzourio-Mazoyer, N. (2015)
#'   Implementation of a new parcellation of the orbitofrontal cortex in the
#'   automated anatomical labelling atlas. \emph{NeuroImage}, \bold{122}, 1--5.
#'   \doi{10.1016/j.neuroimage.2015.07.075}
"aal116"

#' @aliases aal90
#' @rdname brain_atlases
"aal90"

#' @aliases aal2.120
#' @rdname brain_atlases
"aal2.120"

#' @aliases aal2.94
#' @rdname brain_atlases
"aal2.94"

#' Coordinates for data from Freesurfer atlases
#'
#' @aliases destrieux destrieux.scgm dk dk.scgm dkt dkt.scgm
#' @rdname brain_atlases
#' @references Destrieux, C. and Fischl, B. and Dale, A. and Halgren E. (2010)
#'   Automatic parcellation of human cortical gyri and sulci using standard
#'   anatomic nomenclature. \emph{NeuroImage}, \bold{53(1)}, 1--15.
#'   \doi{10.1016/j.neuroimage.2010.06.010}
#' @references Desikan, R.S. and Segonne, F. and Fischl, B. et al. (2006) An
#'   automated labeling system for subdividing the human cerebral cortex on MRI
#'   scans into gyral based regions of interest. \emph{NeuroImage}, \bold{31},
#'   968--980. \doi{10.1016/j.neuroimage.2006.01.021}
#' @references Klein, A. and Tourville, J. (2012) 101 labeled brain images
#'   and a consistent human cortical labeling protocol. \emph{Front Neurosci},
#'   6. \doi{10.3389/fnins.2012.00171}
"destrieux"

#' @rdname brain_atlases
"destrieux.scgm"

#' @rdname brain_atlases
"dk"

#' @rdname brain_atlases
"dk.scgm"

#' @rdname brain_atlases
"dkt"

#' @rdname brain_atlases
"dkt.scgm"

#' Coordinates for data from the Brainsuite atlas
#'
#' @aliases brainsuite
#' @rdname brain_atlases
#' @references Shattuck, D.W. and Leahy, R.M. (2002) BrainSuite: an automated
#'   cortical surface identification tool. \emph{Medical Image Analysis},
#'   \bold{8(2)}, 129--142.
#' @references Pantazis, D. and Joshi, A.A. and Jintao, J. and Shattuck, D.W.
#'   and Bernstein, L.E. and Damasio, H. and Leahy, R.M. (2009) Comparison of
#'   landmark-based and automatic methods for cortical surface registration.
#'   \emph{NeuroImage}, \bold{49(3)}, 2479--2493.
"brainsuite"

#' Coordinates for data from the Craddock200 atlas
#'
#' @aliases craddock200
#' @rdname brain_atlases
#' @references Craddock, R.C. and James, G.A. and Holtzheimer, P.E. and Hu, X.P.
#'   and Mayberg, H.S. (2012) A whole brain fMRI atlas generated via spatially
#'   constrained spectral clustering. \emph{Human Brain Mapping}, \bold{33},
#'   1914--1928. \doi{10.1002/hbm.21333}
"craddock200"

#' Coordinates for data from the Dosenbach160 atlas
#'
#' @aliases dosenbach160
#' @rdname brain_atlases
#' @references Dosenbach, N.U. and Nardos, B. and Cohen, A.L. and Fair, D.A. and
#'   Power, J.D. and Church, J.A. and Nelson, S.M. and Wig, G.S. and Vogel, A.C.
#'   and Lessov-Schlaggar, C.N. and Barnes, K.A. (2010) Prediction of individual
#'   brain maturity using fMRI. \emph{Science}, \bold{329(5997)}, 1358--1361.
"dosenbach160"

#' Coordinates for data from the Harvard-Oxford atlas
#'
#' @aliases hoa112
#' @rdname brain_atlases
#' @references Makris, N. and Goldstein, J.M. and Kennedy, D. et al. (2006)
#'   Decreased volume of left and total anterior insular lobule in
#'   schizophrenia. \emph{Schizophr Res}, \bold{83(2-3)}, 155--171.
"hoa112"

#' Coordinates for data from the LONI probabilistic brain atlas
#'
#' @aliases lpba40
#' @rdname brain_atlases
#' @references Shattuck, D.W. and Mirza, M. and Adisetiyo, V. and Hojatkashani,
#'   C. and Salamon, G. and Narr, K.L. and Poldrack, R.A. and Bilder, R.M. and
#'   Toga, A.W. (2008) Construction of a 3D probabilistic atlas of human
#'   cortical structures. \emph{NeuroImage}, \bold{39(3)}, 1064--1080.
#'   \doi{10.1016/j.neuroimage.2007.09.031}
"lpba40"

#' Coordinates for data from HCP-MMP1.0 atlas
#'
#' @note Use of the HCP parcellation is subject to the terms at
#'   \url{https://balsa.wustl.edu/WN56}. In particular: "I will acknowledge the
#'   use of WU-Minn HCP data and data derived from WU-Minn HCP data when
#'   publicly presenting any results or algorithms that benefitted from their
#'   use."
#'
#' @aliases hcp_mmp1.0
#' @rdname brain_atlases
#' @source \url{https://neuroimaging-core-docs.readthedocs.io/en/latest/pages/atlases.html}
#' @references Glasser, M.F. and Coalson, T.S. and Robinson, E.C. and Hacker,
#'   C.D. and Harwell, J. and Yacoub, E. and Ugurbil, K. and Andersson, J. and
#'   Beckmann, C.F. and Jenkinson, M. and Smith, S.M. and van Essen, D.C.
#'   (2016) A multi-modal parcellation of human cerebral cortex. \emph{Nature},
#'   \bold{536}, 171--178. \doi{10.1038/nature18933}. PMID:
#'   27437579.
"hcp_mmp1.0"

#' Coordinates for data from the Power atlas
#'
#' @aliases power264
#' @rdname brain_atlases
#' @references Power, J.D. and Cohen, A.L. and Nelson, S.M. and Wig, G.S. and
#'   Barnes, K.A. and Church, J.A. and Vogel, A.C. and Laumann, T.O. and Miezin,
#'   F.M. and Schlaggar, B.L. and Petersen, S.E. (2011) Functional network
#'   organization of the human brain. \emph{Neuron}, \bold{72(4)}, 665--678.
#'   \doi{10.1016/j.neuron.2011.09.006}
"power264"

#' Coordinates for data from the Brainnetome atlas
#'
#' @aliases brainnetome
#' @rdname brain_atlases
#' @references Fan, L. and Li, H. and Zhuo, J. and Zhang, Y. and Wang, J. and
#'   Chen, L. and Yang, Z. and Chu, C. and Xie, S. and Laird, A.R. and Fox,
#'   P.T. and Eickhoff, S.B. and Yu, C. and Jiang, T (2016) The Human
#'   Brainnetome Atlas: A New Brain Atlas Based on Connectional Architecture.
#'   \emph{Cerebral Cortex}, \bold{26(8)}, 3508--3526.
#'   \doi{10.1093/cercor/bhw157}

"brainnetome"

#' Coordinates for data from the Gordon atlas
#'
#' @note Region names in the \code{gordon333} atlas were chosen to match those
#'   of the \code{hcp_mmp1.0} atlas. Many were determined from the coordinates
#'   (using FSL's \code{atlasquery}), while the rest were entered manually by
#'   me. The \code{lobe} values were matched to the HCP atlas, as well.
#' @aliases gordon333
#' @rdname brain_atlases
#' @references Gordon, E.M. and Laumann, T.O. and Adeyemo, B. and Huckins, J.F.
#'   and Kelley, W.M. and Petersen, S.E. (2014) Generation and Evaluation of a
#'   Cortical Area Parcellation from Resting-State Correlations. \emph{Cerebral
#'   Cortex}, \bold{26(1)}, 288--303.
#'   \doi{10.1093/cercor/bhu239}
"gordon333"
