% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/data_simulation.R
\name{generatedata_mpin}
\alias{generatedata_mpin}
\title{Simulation of MPIN model data}
\usage{
generatedata_mpin(series = 1, days = 60, layers = NULL,
                         parameters = NULL, ranges = list(), ...,
                         verbose = TRUE)
}
\arguments{
\item{series}{The number of datasets to generate.}

\item{days}{The number of trading days for which aggregated buys and
sells are generated. Default value is \code{60}.}

\item{layers}{The number of information layers to be included in the
simulated data. Default value is \code{NULL}.
If \code{layers} is omitted or set to \code{NULL}, the number of layers is uniformly
selected from the set \code{{1, ..., maxlayers}}.}

\item{parameters}{A vector of model parameters of size \code{3J+2}
where \code{J} is the number of information layers and it has the
following form
\{\ifelse{html}{\eqn{\alpha}\subit{1}}{\eqn{\alpha_1}{}},
...,\ifelse{html}{\eqn{\alpha}\subit{J}}{\eqn{\alpha_J}{}},
\ifelse{html}{\eqn{\delta}\subit{1}}{\eqn{\delta_1}{}},...,
\ifelse{html}{\eqn{\delta}\subit{J}}{\eqn{\delta_J}{}},
\ifelse{html}{\eqn{\mu}\subit{1}}{\eqn{\mu_1}{}},...,
\ifelse{html}{\eqn{\mu}\subit{J}}{\eqn{\mu_J}{}}, \eb, \es\}.}

\item{ranges}{A list of ranges for the different simulation
parameters having named elements \eqn{\alpha}, \eqn{\delta}, \eb, \es,
and \eqn{\mu}.
The value of each element is a vector of two numbers: the first one is the
minimal value \code{min_v} and the second one is the maximal value \code{max_v}.
If the element corresponding to a given parameter is missing, the default
range for that parameter is used. If the argument \code{ranges} is an empty
list and \code{parameters} is \code{NULL}, the default ranges for the parameters
are used. The simulation parameters are uniformly drawn from the interval
(\code{min_v}, \code{max_v}) for the specified parameters.
The default value is \code{list()}.}

\item{...}{Additional arguments passed on to the function
\code{generatedata_mpin()}. The recognized arguments are \code{confidence},
\code{maxlayers}, \code{eps_ratio}, \code{mu_ratio}.
\itemize{
\item \code{confidence} (\code{numeric}) denotes the range of the confidence interval
associated with each layer such that all observations within the layer \code{j}
lie in the theoretical confidence interval of the Skellam distribution
centered on the mean order imbalance, at the level \code{'confidence'}.
The default value is \code{0.99}.
\item \code{maxlayers} (\code{integer}) denotes the upper limit of number of layers
for the generated datasets. If the argument \code{layers} is missing, the layers
of the simulated datasets will be uniformly drawn from
\code{{1,..., maxlayers}}. When missing, \code{maxlayers} takes the default
value of \code{5}.
\item \code{eps_ratio} (\code{numeric}) specifies the admissible range for the value
of the ratio \es/\eb, It can be a two-value vector or just a single value.
If \code{eps_ratio} is a vector of two values: the first one is the minimal value
and the second one is the maximal value; and the function tries to generate
\es and \eb satisfying that their ratios \es/\eb lies within the interval
\code{eps_ratio}. If \code{eps_ratio} is a single number, then the function tries to
generate \es and \eb satisfying \es = \eb x \code{eps_ratio}. If this range
conflicts with other arguments such as \code{ranges}, a warning is displayed.
The default value is \code{c(0.75, 1.25)}.
\item \code{mu_ratio} (\code{numeric}) it is the minimal value of the ratio between
two consecutive values of the vector \code{mu}. If \code{mu_ratio = 1.25} e.g., then
\ifelse{html}{\eqn{\mu}\subit{j+1}}{\eqn{\mu_{j+1}}{}} should be larger than
\code{1.25}* \ifelse{html}{\eqn{\mu}\subit{j}}{\eqn{\mu_{j}}{}} for all
\verb{j = 1, .., J}. If \code{mu_ratio} conflicts with other arguments such as \code{ranges}
or \code{confidence}, a warning is displayed. The default value is \code{NULL}.
}}

\item{verbose}{(\code{logical}) a binary variable that determines whether detailed
information about the progress of the data generation is displayed.
No output is produced when \code{verbose} is set to \code{FALSE}. The default
value is \code{TRUE}.}
}
\value{
Returns an object of class \link{dataset-class} if \code{series=1}, and an
object of class \link{data.series-class} if \code{series>1}.
}
\description{
Generates a \code{dataset} object or a \code{data.series} object (a list
of \code{dataset} objects) storing simulation parameters as well as aggregate
daily buys and sells simulated following the assumption of the \code{MPIN} model
of \insertCite{Ersan2016}{PINstimation}.
}
\details{
An information layer refers to a given type of information event existing
in the data. The \code{PIN} model assumes a single type of information events
characterized by three parameters for \eqn{\alpha}, \eqn{\delta}, and
\eqn{\mu}. The \code{MPIN} model relaxes the assumption, by relinquishing the
restriction on the number of information event types. When \code{layers = 1},
generated data fit the assumptions of the \code{PIN} model.

If the argument \code{parameters} is missing, then the simulation parameters are
generated using the ranges specified in the argument \code{ranges}.
If the argument \code{ranges} is \code{list()}, default ranges are used. Using the
default ranges, the simulation parameters are obtained using the following
procedure:
\itemize{
\item \eqn{\alpha()}: a vector of length \code{layers}, where each
\ifelse{html}{\eqn{\alpha}\subit{j}}{\eqn{\alpha_j}{}} is uniformly
distributed on \code{(0, 1)} subject to the condition:
\ifelse{html}{\eqn{\sum \alpha}\subit{j}\eqn{< 1}}{
\eqn{\sum_j \alpha_j \leq1}{}}.
\item \eqn{\delta()}: a vector of length \code{layers}, where each
\ifelse{html}{\eqn{\delta}\subit{j}}{\eqn{\delta_j}{}} uniformly distributed
on \code{(0, 1)}.
\item \eqn{\mu()}: a vector of length \code{layers}, where each
\ifelse{html}{\eqn{\mu}\subit{j}}{\eqn{\mu_j}{}} is uniformly distributed
on the interval \verb{(0.5 max(}\eb\verb{,} \es\verb{), 5 max(}\eb\verb{,} \es\verb{))}.
The \eqn{\mu}:s are then sorted so the excess trading increases in the
information layers, subject to the condition that the ratio of two
consecutive \eqn{\mu}'s should be at least \code{1.25}.
\item \eb: an integer drawn uniformly from the interval \code{(100, 10000)}
with step \code{50}.
\item \es: an integer uniformly drawn from (\code{(3/4)}\eb, \code{(5/4)}\eb) with step
\code{50}.
}
Based on the simulation parameters \code{parameters}, daily buys and sells are
generated by the assumption that buys and sells
follow Poisson distributions with mean parameters (\eb, \es) on days with no
information; with mean parameters
(\eb + \ifelse{html}{\eqn{\mu}\subit{j}}{ \eqn{\mu_j}{}}, \es) on days
with good information of layer \eqn{j} and
(\eb, \es + \ifelse{html}{\eqn{\mu}\subit{j}}{\eqn{\mu_j}{}}) on days
with bad information of layer \eqn{j}.

\strong{Considerations for the ranges of simulation parameters:} While
\code{generatedata_mpin()} function enables the user to simulate data series
with any set of theoretical parameters,
we strongly recommend the use of parameter sets satisfying below conditions
which are in line with the nature of empirical data and the theoretical
models used within this package.
When parameter values are not assigned by the user, the function, by default,
simulates data series that are in line with these criteria.
\itemize{
\item \emph{Consideration 1}: any \eqn{\mu}'s value separable from \eb and \es
values, as well as other \eqn{\mu} values. Otherwise, the \code{PIN} and \code{MPIN}
estimation would not yield expected results.\cr
[x] Sharp example.1: \eb\eqn{ = 1000}; \eqn{\mu = 1}. In this case, no
information layer can be captured in a healthy way by the use of the models
which relies on Poisson distributions.\cr
[x] Sharp example.2: \es\eqn{ = 1000},
\ifelse{html}{\eqn{\mu}\subit{1}\eqn{ = 1000}}{\eqn{\mu_1 = 1000}{}},
and \ifelse{html}{\eqn{\mu}\subit{2}\eqn{ = 1001}}{\eqn{\mu_2 = 1001}{}}.
Similarly, no distinction can be
made on the two simulated layers of informed trading. In real life, this
entails that there is only one type of information which would also be the
estimate of the \code{MPIN} model. However, in the simulated data properties,
there would be 2 layers which will lead the user to make a wrong
evaluation of model performance.

\item \emph{Consideration 2}: \eb and \es being relatively close to each other.
When they are far from each other, that would indicate that there is
substantial asymmetry between buyer and seller initiated trades, being a
strong signal for informed trading.
There is no theoretical evidence to indicate that the uninformed trading in
buy and sell sides deviate much from each other in real life.
Besides, numerous papers that work with \code{PIN} model provide close to
each other uninformed intensities.
when no parameter values are assigned by the user, the function generates
data with the condition of sell side uninformed trading to be in the range of
\verb{(4/5):=80\%} and \verb{(6/5):=120\%} of buy side uninformed rate.\cr
[x] Sharp example.3: \eb\eqn{ = 1000}, \es\eqn{ = 10000}. In this
case, the \code{PIN} and \code{MPIN} models would tend to consider some of the trading
in sell side to be informed (which should be the actual case).
Again, the estimation results would deviate much from the simulation
parameters being a good news by itself but a misleading factor in model
evaluation.
See for example \insertCite{ChengLai2021;textual}{PINstimation} as a
misinterpretation of comparative performances. The paper's findings highly
rely on the simulations with extremely different \eb and \es values
(813-8124 pair and 8126-812).
}
}
\examples{
# ------------------------------------------------------------------------ #
# There are different scenarios of using the function generatedata_mpin()  #
# ------------------------------------------------------------------------ #

# With no arguments, the function generates one dataset object spanning
# 60 days, containing a number of information layers uniformly selected
# from `{1, 2, 3, 4, 5}`, and where the parameters are chosen as
# described in the details.

sdata <- generatedata_mpin()

# The number of layers can be deduced from the simulation parameters, if
# fed directly to the function generatedata_mpin() through the argument
# 'parameters'. In this case, the output is a dataset object with one
# information layer.

givenpoint <- c(0.4, 0.1, 800, 300, 200)
sdata <- generatedata_mpin(parameters = givenpoint)

# The number of layers can alternatively be set directly through the
# argument 'layers'.

sdata <- generatedata_mpin(layers = 2)

# The simulation parameters can be randomly drawn from their corresponding
# ranges fed through the argument 'ranges'.

sdata <- generatedata_mpin(ranges = list(alpha = c(0.1, 0.7),
                                        delta = c(0.2, 0.7),
                                        mu = c(3000, 5000)))

# The value of a given simulation parameter can be set to a specific value by
# setting the range of the desired parameter takes a unique value, instead of
# a pair of values.

sdata <- generatedata_mpin(ranges = list(alpha = 0.4, delta = c(0.2, 0.7),
                                        eps.b = c(100, 7000),
                                        mu = c(8000, 12000)))

# If both arguments 'parameters', and 'layers' are simultaneously provided,
# and the number of layers detected from the length of the argument
# 'parameters' is different from the argument 'layers', the former is used
# and a warning is displayed.

sim.params <- c(0.4, 0.2, 0.9, 0.1, 400, 700, 300, 200)
sdata <- generatedata_mpin(days = 120, layers = 3, parameters = sim.params)

# Display the details of the generated data

show(sdata)
\donttest{
# ------------------------------------------------------------------------ #
# Use generatedata_mpin() to compare the accuracy of estimation methods    #
# ------------------------------------------------------------------------ #

# The example below illustrates the use of the function 'generatedata_mpin()'
# to compare the accuracy of the functions 'mpin_ml()', and 'mpin_ecm()'.

# The example will depend on three variables:
# n: the number of datasets used
# l: the number of layers in each simulated datasets
# xc : the number of extra clusters used in initials_mpin

# For consideration of speed, we will set n = 2, l = 2, and xc = 2
# These numbers can change to fit the user's preferences
n <- l <- xc <- 2

# We start by generating n datasets simulated according to the
# assumptions of the MPIN model.

dataseries <- generatedata_mpin(series = n, layers = l, verbose = FALSE)

# Store the estimates in two different lists: 'mllist', and 'ecmlist'

mllist <- lapply(dataseries@datasets, function(x)
  mpin_ml(x@data, xtraclusters = xc, layers = l, verbose = FALSE))

ecmlist <- lapply(dataseries@datasets, function(x)
  mpin_ecm(x@data, xtraclusters = xc, layers = l, verbose = FALSE))

# For each estimate, we calculate the absolute difference between the
# estimated mpin, and empirical mpin computed using dataset parameters.
# The absolute differences are stored in 'mldmpin' ('ecmdpin') for the
# ML (ECM) method,

mldpin <- sapply(1:n,
 function(x) abs(mllist[[x]]@mpin - dataseries@datasets[[x]]@emp.pin))

ecmdpin <- sapply(1:n,
 function(x) abs(ecmlist[[x]]@mpin - dataseries@datasets[[x]]@emp.pin))

# Similarly, we obtain vectors of running times for both estimation methods.
# They are stored in 'mltime' ('ecmtime') for the ML (ECM) method.

mltime <- sapply(mllist, function(x) x@runningtime)
ecmtime <- sapply(ecmlist, function(x) x@runningtime)

# Finally, we calculate the average absolute deviation from empirical PIN
# as well as the average running time for both methods. This allows us to
# compare them in terms of accuracy, and speed.

accuracy <- c(mean(mldpin), mean(ecmdpin))
timing <- c(mean(mltime), mean(ecmtime))
comparison <- as.data.frame(rbind(accuracy, timing))
colnames(comparison) <- c("ML", "ECM")
rownames(comparison) <- c("Accuracy", "Timing")

show(round(comparison, 6))
}
}
\references{
\insertAllCited
}
