#' @title Distributed Probabilistic Principal Component Analysis
#' @description Performs distributed probabilistic principal component analysis (PPC)
#' on a numeric dataset split across multiple nodes. Estimates loading matrices,
#' residual variances, and covariance matrices for each node using a probabilistic approach.
#'
#' @param data A numeric matrix containing the total dataset.
#' @param m An integer specifying the number of principal components.
#' @param n1 An integer specifying the length of each data subset.
#' @param K An integer specifying the number of nodes.
#'
#' @return A list with the following components:
#' \describe{
#'   \item{Apro}{List of estimated loading matrices for each node.}
#'   \item{Dpro}{List of diagonal residual variance matrices for each node.}
#'   \item{Sigmahatpro}{List of covariance matrices for each node.}
#' }
#'
#' @examples
#' set.seed(123)
#' data <- matrix(rnorm(500), nrow = 100, ncol = 5)
#' DPPC(data = data, m = 3, n1 = 20, K = 5)
#'
#' @export
DPPC <- function(data, m, n1, K) {
  n <- nrow(data)
  p <- ncol(data)
  P <- as.matrix(diag(c(0, 1), n, n))
  X1pro <- scale(P %*% as.matrix(data))
  Sigmahatpro <- list()
  Apro <- list()
  Dpro <- list()
  
  for (i in 1:K) {
    L <- matrix(rep(0, K * n1), ncol = n1)
    R <- matrix(0, n1, n)
    L[i, ] <- sample(1:n, n1, replace = FALSE)
    r <- matrix(c(1:n1, L[i, ]), ncol = n1, byrow = TRUE)
    R[t(r)] <- 1
    X1 <- R %*% X1pro
    Xpro <- scale(X1)
    
    Sigmahatpro[[i]] <- cor(Xpro)
    eig1 <- eigen(Sigmahatpro[[i]])
    lambdahat <- eig1$values[1:m]
    ind <- order(lambdahat, decreasing = TRUE)
    lambdahat <- lambdahat[ind]
    Q <- eig1$vectors[, ind]
    Qhat <- Q[, 1:m]
    
    Apro1 <- matrix(0, nrow = p, ncol = m)
    for (j in 1:m) {
      Apro1[, j] <- sqrt(lambdahat[j]) * Qhat[, j]
    }
    Apro[[i]] <- Apro1
    hpro <- diag(Apro[[i]] %*% t(Apro[[i]]))
    Dpro[[i]] <- diag(Sigmahatpro[[i]] - hpro)
  }
  
  return(list(Apro = Apro, Dpro = Dpro, Sigmahatpro = Sigmahatpro))
}