% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/classes__OAuthClient.R
\name{oauth_client}
\alias{oauth_client}
\title{Create generic \link{OAuthClient}}
\usage{
oauth_client(
  provider,
  client_id = Sys.getenv("OAUTH_CLIENT_ID"),
  client_secret = Sys.getenv("OAUTH_CLIENT_SECRET"),
  redirect_uri,
  scopes = character(0),
  state_store = cachem::cache_mem(max_age = 300),
  state_entropy = 64,
  state_key = random_urlsafe(128),
  client_private_key = NULL,
  client_private_key_kid = NULL,
  client_assertion_alg = NULL
)
}
\arguments{
\item{provider}{\link{OAuthProvider} object}

\item{client_id}{OAuth client ID}

\item{client_secret}{OAuth client secret.

Validation rules:
\itemize{
\item Required (non-empty) when the provider authenticates the client with
HTTP Basic auth at the token endpoint (\code{token_auth_style = "header"},
also known as \code{client_secret_basic}).
\item Optional for public PKCE-only clients when the provider is configured
with \code{use_pkce = TRUE} and uses form-body client authentication at the
token endpoint (\code{token_auth_style = "body"}, also known as
\code{client_secret_post}). In this case, the secret is omitted from token
requests.
}

Note: If your provider issues HS256 ID tokens and \code{id_token_validation} is
enabled, a non-empty \code{client_secret} is required for signature validation.}

\item{redirect_uri}{Redirect URI registered with provider}

\item{scopes}{Vector of scopes to request}

\item{state_store}{State storage backend. Defaults to \code{cachem::cache_mem(max_age = 300)}.
Alternative backends could include \code{cachem::cache_disk()} or a custom
implementation (which you can create with \code{\link[=custom_cache]{custom_cache()}}. The backend
must implement cachem-like methods \verb{$get(key, missing)}, \verb{$set(key, value)},
and \verb{$remove(key)}; \verb{$info()} is optional.

Trade-offs: \code{cache_mem} is in-memory and thus scoped to a single R process
(good default for a single Shiny process). \code{cache_disk} persists to disk
and can be shared across multiple R processes (useful for multi-process
deployments or when Shiny workers aren't sticky). A \code{\link[=custom_cache]{custom_cache()}}
backend could use a database or external store (e.g., Redis, Memcached).
See also \code{vignette("usage", package = "shinyOAuth")}.

The client automatically generates, persists (in \code{state_store}), and
validates the OAuth \code{state} parameter (and OIDC \code{nonce} when applicable)
during the authorization code flow}

\item{state_entropy}{Integer. The length (in characters) of the randomly
generated state parameter. Higher values provide more entropy and better
security against CSRF attacks. Must be between 22 and 128 (to align with
\code{validate_state()}'s default minimum which targets ~128 bits for base64url‑like
strings). Default is 64, which provides approximately 384 bits of entropy}

\item{state_key}{Optional per-client secret used as the state sealing key
for AES-GCM AEAD (authenticated encryption) of the state payload that
travels via the \code{state} query parameter. This provides confidentiality
and integrity (via authentication tag) for the embedded data used during
callback verification. If you omit this argument, a random value is
generated via \code{random_urlsafe(128)}. This key is distinct from the
OAuth \code{client_secret} and may be used with public clients.

Type: character string (>= 32 bytes when encoded) or raw vector
(>= 32 bytes). Raw keys enable direct use of high-entropy secrets from
external stores. Both forms are normalized internally by cryptographic
helpers.

Multi-process deployments: if your app runs with multiple R workers or behind
a non-sticky load balancer, you must configure a shared \code{state_store} and the
same \code{state_key} across all workers. Otherwise callbacks that land on a
different worker will be unable to decrypt/validate the state envelope and
authentication will fail. In such environments, do not rely on the random
per-process default: provide an explicit, high-entropy key (for example via
a secret store or environment variable). Prefer values with substantial
entropy (e.g., 64–128 base64url characters or a raw 32+ byte key). Avoid
human‑memorable passphrases. See also \code{vignette("usage", package = "shinyOAuth")}.}

\item{client_private_key}{Optional private key for \code{private_key_jwt} client authentication
at the token endpoint. Can be an \code{openssl::key} or a PEM string containing a
private key. Required when the provider's \code{token_auth_style = 'private_key_jwt'}.
Ignored for other auth styles.}

\item{client_private_key_kid}{Optional key identifier (kid) to include in the JWT header
for \code{private_key_jwt} assertions. Useful when the authorization server uses kid to
select the correct verification key.}

\item{client_assertion_alg}{Optional JWT signing algorithm to use for client assertions.
When omitted, defaults to \code{HS256} for \code{client_secret_jwt}. For \code{private_key_jwt}, a
compatible default is selected based on the private key type/curve (e.g., \code{RS256} for RSA,
\code{ES256}/\code{ES384}/\code{ES512} for EC P-256/384/521, or \code{EdDSA} for Ed25519/Ed448). If an explicit
value is provided but incompatible with the key, validation fails early with a configuration
error.
Supported values are \code{HS256}, \code{HS384}, \code{HS512} for client_secret_jwt and asymmetric algorithms
supported by \code{jose::jwt_encode_sig} (e.g., \code{RS256}, \code{PS256}, \code{ES256}, \code{EdDSA}) for private keys.}
}
\value{
\link{OAuthClient} object
}
\description{
Create generic \link{OAuthClient}
}
\examples{
if (
  # Example requires configured GitHub OAuth 2.0 app
  # (go to https://github.com/settings/developers to create one):
  nzchar(Sys.getenv("GITHUB_OAUTH_CLIENT_ID")) 
  && nzchar(Sys.getenv("GITHUB_OAUTH_CLIENT_SECRET"))
  && interactive()
) {
  library(shiny)
  library(shinyOAuth)
  
  # Define client
  client <- oauth_client(
    provider = oauth_provider_github(),
    client_id = Sys.getenv("GITHUB_OAUTH_CLIENT_ID"),
    client_secret = Sys.getenv("GITHUB_OAUTH_CLIENT_SECRET"),
    redirect_uri = "http://127.0.0.1:8100"
  )
  
  # Choose which app you want to run
  app_to_run <- NULL
  while (!isTRUE(app_to_run \%in\% c(1:4))) {
    app_to_run <- readline(
      prompt = paste0(
        "Which example app do you want to run?\n",
        "  1: Auto-redirect login\n",
        "  2: Manual login button\n",
        "  3: Fetch additional resource with access token\n",
        "  4: No app (all will be defined but none run)\n",
        "Enter 1, 2, 3, or 4... "
      )
    )
  }
  
  
  # Example app with auto-redirect (1) -----------------------------------------
  
  ui_1 <- fluidPage(
    use_shinyOAuth(),
    uiOutput("login")
  )
  
  server_1 <- function(input, output, session) {
    # Auto-redirect (default):
    auth <- oauth_module_server(
      "auth",
      client,
      auto_redirect = TRUE
    )
    
    output$login <- renderUI({
      if (auth$authenticated) {
        user_info <- auth$token@userinfo
        tagList(
          tags$p("You are logged in!"),
          tags$pre(paste(capture.output(str(user_info)), collapse = "\n"))
        )
      } else {
        tags$p("You are not logged in.")
      }
    })
  }
  
  app_1 <- shinyApp(ui_1, server_1)
  if (app_to_run == "1") {
    runApp(app_1, port = 8100)
  }
  
  
  # Example app with manual login button (2) -----------------------------------
  
  ui_2 <- fluidPage(
    use_shinyOAuth(),
    actionButton("login_btn", "Login"),
    uiOutput("login")
  )
  
  server_2 <- function(input, output, session) {
    auth <- oauth_module_server(
      "auth",
      client,
      auto_redirect = FALSE
    )
    
    observeEvent(input$login_btn, {
      auth$request_login()
    })
    
    output$login <- renderUI({
      if (auth$authenticated) {
        user_info <- auth$token@userinfo
        tagList(
          tags$p("You are logged in!"),
          tags$pre(paste(capture.output(str(user_info)), collapse = "\n"))
        )
      } else {
        tags$p("You are not logged in.")
      }
    })
  }
  
  app_2 <- shinyApp(ui_2, server_2)
  if (app_to_run == "2") {
    runApp(app_2, port = 8100)
  }
  

  # Example app requesting additional resource with access token (3) -----------
  
  # Below app shows the authenticated username + their GitHub repositories,
  # fetched via GitHub API using the access token obtained during login
  
  ui_3 <- fluidPage(
    use_shinyOAuth(),
    uiOutput("ui")
  )
  
  server_3 <- function(input, output, session) {
    auth <- oauth_module_server(
      "auth",
      client,
      auto_redirect = TRUE
    )
    
    repositories <- reactiveVal(NULL)
    
    observe({
      req(auth$authenticated)
      
      # Example additional API request using the access token
      # (e.g., fetch user repositories from GitHub)
      req <- client_bearer_req(auth$token, "https://api.github.com/user/repos")
      resp <- httr2::req_perform(req)
      
      if (httr2::resp_is_error(resp)) {
        repositories(NULL)
      } else {
        repos_data <- httr2::resp_body_json(resp, simplifyVector = TRUE)
        repositories(repos_data)
      }
    })
    
    # Render username + their repositories
    output$ui <- renderUI({
      if (isTRUE(auth$authenticated)) {
        user_info <- auth$token@userinfo
        repos <- repositories()
        
        return(tagList(
          tags$p(paste("You are logged in as:", user_info$login)),
          tags$h4("Your repositories:"),
          if (!is.null(repos)) {
            tags$ul(
              Map(function(url, name) {
                tags$li(tags$a(href = url, target = "_blank", name))
              }, repos$html_url, repos$full_name)
            )
          } else {
            tags$p("Loading repositories...")
          }
        ))
      }
      
      return(tags$p("You are not logged in."))
    })
  }
  
  app_3 <- shinyApp(ui_3, server_3)
  if (app_to_run == "3") {
    runApp(app_3, port = 8100)
  }
}
}
