% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bundle_torch.R
\name{bundle.luz_module_fitted}
\alias{bundle.luz_module_fitted}
\title{Bundle a \code{luz_module_fitted} object}
\usage{
\method{bundle}{luz_module_fitted}(x, ...)
}
\arguments{
\item{x}{A \code{luz_module_fitted} object returned from
\code{\link[luz:fit.luz_module_generator]{luz::fit.luz_module_generator()}}.}

\item{...}{Not used in this bundler and included for compatibility with
the generic only. Additional arguments passed to this method will return
an error.}
}
\value{
A bundle object with subclass \code{bundled_luz_module_fitted}.

Bundles are a list subclass with two components:

\item{object}{An R object. Gives the output of native serialization
methods from the model-supplying package, sometimes with additional
classes or attributes that aid portability. This is often
a \link[base:raw]{raw} object.}
\item{situate}{A function. The \code{situate()} function is defined when
\code{\link[=bundle]{bundle()}} is called, though is a loose analogue of an \code{\link[=unbundle]{unbundle()}} S3
method for that object. Since the function is defined on \code{\link[=bundle]{bundle()}}, it
has access to references and dependency information that can
be saved alongside the \code{object} component. Calling \code{\link[=unbundle]{unbundle()}} on a
bundled object \code{x} calls \code{x$situate(x$object)}, returning the
unserialized version of \code{object}. \code{situate()} will also restore needed
references, such as server instances and environmental variables.}

Bundles are R objects that represent a "standalone" version of their
analogous model object. Thus, bundles are ready for saving to a file; saving
with \code{\link[base:readRDS]{base::saveRDS()}} is our recommended serialization strategy for bundles,
unless documented otherwise for a specific method.

To restore the original model object \code{x} in a new environment, load its
bundle with \code{\link[base:readRDS]{base::readRDS()}} and run \code{\link[=unbundle]{unbundle()}} on it. The output
of \code{\link[=unbundle]{unbundle()}} is a model object that is ready to \code{\link[=predict]{predict()}} on new data,
and other restored functionality (like plotting or summarizing) is supported
as a side effect only.

The bundle package wraps native serialization methods from model-supplying
packages. Between versions, those model-supplying packages may change their
native serialization methods, possibly introducing problems with re-loading
objects serialized with previous package versions. The bundle package does
not provide checks for these sorts of changes, and ought to be used in
conjunction with tooling for managing and monitoring model environments
like \link[vetiver:vetiver-package]{vetiver} or \link[renv:renv-package]{renv}.

See \code{vignette("bundle")} for more information on bundling and its motivation.
}
\description{
Bundling a model prepares it to be saved to a file and later
restored for prediction in a new R session. See the 'Value' section for
more information on bundles and their usage.
}
\details{
For now, bundling methods for torch are only available
via the luz package, "a higher level API for torch providing
abstractions to allow for much less verbose training loops."
}
\examples{
\dontshow{if (rlang::is_installed(c("torch")) && identical(Sys.getenv("NOT_CRAN"), "true")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
if (torch::torch_is_installed()) {
# fit model and bundle ------------------------------------------------
library(torch)
library(torchvision)
library(luz)

set.seed(1)

# example adapted from luz pkgdown article "Autoencoder"
dir <- tempdir()

mnist_dataset2 <- torch::dataset(
  inherit = mnist_dataset,
  .getitem = function(i) {
    output <- super$.getitem(i)
    output$y <- output$x
    output
  }
)

train_ds <- mnist_dataset2(
  dir,
  download = TRUE,
  transform = transform_to_tensor
)

test_ds <- mnist_dataset2(
  dir,
  train = FALSE,
  transform = transform_to_tensor
)

train_dl <- dataloader(train_ds, batch_size = 128, shuffle = TRUE)
test_dl <- dataloader(test_ds, batch_size = 128)

net <- nn_module(
  "Net",
  initialize = function() {
    self$encoder <- nn_sequential(
      nn_conv2d(1, 6, kernel_size=5),
      nn_relu(),
      nn_conv2d(6, 16, kernel_size=5),
      nn_relu()
    )
    self$decoder <- nn_sequential(
      nn_conv_transpose2d(16, 6, kernel_size = 5),
      nn_relu(),
      nn_conv_transpose2d(6, 1, kernel_size = 5),
      nn_sigmoid()
    )
  },
  forward = function(x) {
    x \%>\%
      self$encoder() \%>\%
      self$decoder()
  },
  predict = function(x) {
    self$encoder(x) \%>\%
      torch_flatten(start_dim = 2)
  }
)

mod <- net \%>\%
  setup(
    loss = nn_mse_loss(),
    optimizer = optim_adam
  ) \%>\%
  fit(train_dl, epochs = 1, valid_data = test_dl)

mod_bundle <- bundle(mod)


# then, after saveRDS + readRDS or passing to a new session ----------
mod_unbundled <- unbundle(mod_bundle)

mod_unbundled_preds <- predict(mod_unbundled, test_dl)
}
\dontshow{\}) # examplesIf}
}
\seealso{
This method wraps \code{\link[luz:luz_save]{luz::luz_save()}} and \code{\link[luz:luz_load]{luz::luz_load()}}.

Other bundlers: 
\code{\link{bundle}()},
\code{\link{bundle.H2OAutoML}()},
\code{\link{bundle.bart}()},
\code{\link{bundle.keras.engine.training.Model}()},
\code{\link{bundle.model_fit}()},
\code{\link{bundle.model_stack}()},
\code{\link{bundle.recipe}()},
\code{\link{bundle.step_umap}()},
\code{\link{bundle.train}()},
\code{\link{bundle.workflow}()},
\code{\link{bundle.xgb.Booster}()}
}
\concept{bundlers}
