% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/02-vegetation-indices.R
\name{analyze_crop_vegetation}
\alias{analyze_crop_vegetation}
\title{Specialized crop vegetation analysis}
\usage{
analyze_crop_vegetation(
  spectral_data,
  crop_type = "general",
  growth_stage = "unknown",
  analysis_type = "comprehensive",
  cdl_mask = NULL,
  reference_data = NULL,
  output_folder = NULL,
  verbose = FALSE
)
}
\arguments{
\item{spectral_data}{Multi-band spectral data (file, directory, or SpatRaster)}

\item{crop_type}{Crop type for specialized analysis ("corn", "soybeans", "wheat", "general")}

\item{growth_stage}{Growth stage if known ("early", "mid", "late", "harvest")}

\item{analysis_type}{Type of analysis: "comprehensive", "stress", "growth", "yield"}

\item{cdl_mask}{Optional CDL mask for crop-specific analysis}

\item{reference_data}{Optional reference data for validation}

\item{output_folder}{Optional output folder for results}

\item{verbose}{Print detailed progress}
}
\value{
List with comprehensive vegetation analysis results:
\itemize{
\item \code{vegetation_indices}: SpatRaster with calculated indices
\item \code{analysis_results}: Detailed analysis results by type
\item \code{metadata}: Analysis metadata and parameters
}
}
\description{
Perform comprehensive vegetation analysis specifically designed for crop monitoring
including growth stage detection, stress identification, and yield prediction support.
Handles test scenarios properly with better input validation.
}
\details{
\subsection{Crop-Specific Index Selection:}{
\itemize{
\item \strong{Corn}: NDVI, EVI, GNDVI, DVI, RVI, PRI
\item \strong{Soybeans}: NDVI, EVI, SAVI, GNDVI, PRI
\item \strong{Wheat}: NDVI, EVI, SAVI, DVI
\item \strong{General}: NDVI, EVI, SAVI, GNDVI, DVI, RVI
}
}

\subsection{Analysis Types:}{
\itemize{
\item \strong{comprehensive}: All analyses (stress, growth, yield)
\item \strong{stress}: Focus on stress detection indices
\item \strong{growth}: Growth stage analysis
\item \strong{yield}: Yield prediction support
}
}

\subsection{Output Structure:}{

The function returns a list with three main components:
\subsection{1. vegetation_indices (SpatRaster)}{

Multi-layer raster with calculated indices (NDVI, EVI, etc.)
}

\subsection{2. analysis_results (List)}{

\strong{stress_analysis} (if requested):
\itemize{
\item Percentage of pixels in each stress category
\item Categories: healthy (NDVI 0.6-1.0), moderate stress (0.4-0.6), severe stress (0.0-0.4)
\item Includes mean, median, std_dev, and thresholds used
}

\strong{growth_analysis} (if requested):
\itemize{
\item Predicted growth stage based on NDVI patterns
\item Stage confidence (0-1 scale)
\item Detailed statistics for each index
\item Growth stages: emergence, vegetative, reproductive, maturity (crop-specific)
}

\strong{yield_analysis} (if requested):
\itemize{
\item \strong{Composite Yield Index}: Normalized 0-1 score combining multiple indices
\itemize{
\item 0.0 = Very low yield potential
\item 0.5 = Medium yield potential
\item 1.0 = Maximum yield potential
}
\item \strong{Yield Potential Class}: Categorical (Low, Medium, High, Very High)
\item \strong{Index Contributions}: How each index contributed to composite score
\item Calculation: Each index (NDVI, EVI, GNDVI, DVI, RVI) is normalized to 0-1,
then averaged to create composite score
}

\strong{summary_statistics}:
\itemize{
\item Basic stats (mean, std, min, max, percentiles) for all indices
\item Coverage percentage and pixel counts
}
}

\subsection{3. metadata (List)}{

Processing information: crop_type, indices_used, processing_date, spatial properties
}

}

\subsection{Example Interpretation:}{

\if{html}{\out{<div class="sourceCode r">}}\preformatted{result <- analyze_crop_vegetation(data, crop_type = "corn")

# Stress Assessment
stress <- result$analysis_results$stress_analysis$NDVI
cat(sprintf("Healthy: \%.1f\%\%, Stressed: \%.1f\%\%\\n",
            stress$healthy_percentage,
            stress$severe_stress_percentage))

# Growth Stage
stage <- result$analysis_results$growth_analysis$predicted_growth_stage
cat(sprintf("Growth stage: \%s\\n", stage))

# Yield Potential
yield <- result$analysis_results$yield_analysis
cat(sprintf("Yield potential: \%s (score: \%.2f)\\n",
            yield$yield_potential_class,
            yield$composite_yield_index))
}\if{html}{\out{</div>}}
}

\subsection{Important Notes:}{
\itemize{
\item \strong{Composite Yield Index} is a vegetation-based proxy, not a direct yield prediction
\item Thresholds are based on literature and may need regional calibration
\item Results should be validated with ground truth data
\item For detailed output documentation, see package vignette
}
}

\subsection{Analysis Types:}{
\itemize{
\item \strong{comprehensive}: All analyses (stress, growth, yield)
\item \strong{stress}: Focus on stress detection indices
\item \strong{growth}: Growth stage analysis
\item \strong{yield}: Yield prediction support
}
}
}
\examples{
\dontrun{
# These examples require actual spectral data
# Comprehensive corn analysis
corn_analysis <- analyze_crop_vegetation(
  spectral_data = sentinel_data,
  crop_type = "corn",
  analysis_type = "comprehensive",
  cdl_mask = corn_mask
)

# Access results
corn_analysis$vegetation_indices        # SpatRaster with indices
corn_analysis$analysis_results$stress_analysis  # Stress detection results
corn_analysis$metadata$indices_used     # Which indices were calculated

# Stress detection in soybeans
stress_analysis <- analyze_crop_vegetation(
  spectral_data = landsat_stack,
  crop_type = "soybeans",
  analysis_type = "stress",
  growth_stage = "mid"
)
}

\donttest{
# Example with mock spectral data
# Create mock multi-band raster (simulating satellite data)
red_band <- terra::rast(nrows = 5, ncols = 5, crs = "EPSG:4326")
nir_band <- terra::rast(nrows = 5, ncols = 5, crs = "EPSG:4326")
terra::values(red_band) <- runif(25, 0.1, 0.3)   # Typical red values
terra::values(nir_band) <- runif(25, 0.4, 0.8)   # Typical NIR values
spectral_stack <- c(red_band, nir_band)
names(spectral_stack) <- c("red", "nir")

# Analyze with mock data
result <- analyze_crop_vegetation(spectral_stack, crop_type = "general")
print(names(result))  # Should show analysis components
}

}
