
#' R6 Class Representing PriorBVARPANEL
#'
#' @description
#' The class PriorBVARPANEL presents a prior specification for the Bayesian
#' hierarchical panel VAR model.
#' 
#' @examples 
#' prior = specify_prior_bvarPANEL$new(C = 2, N = 3, p = 1)
#' prior$M
#' 
#' @export
specify_prior_bvarPANEL = R6::R6Class(
  "PriorBVARPANEL",
  
  public = list(
    
    #' @field M an \code{KxN} matrix, the mean of the second-level MNIW prior
    #' distribution for the global parameter matrices \eqn{\mathbf{A}} 
    #' and \eqn{\mathbf{V}}
    M           = matrix(),
    
    #' @field W a \code{KxK} column-specific covariance matrix of the second-level
    #' MNIW prior distribution for the global parameter matrices \eqn{\mathbf{A}}
    #' and \eqn{\mathbf{V}}
    W           = matrix(),
    
    #' @field S_inv an \code{NxN} row-specific precision matrix of the second-level
    #' MNIW prior distribution for the global parameter matrices \eqn{\mathbf{A}}
    #' and \eqn{\mathbf{V}}
    S_inv       = matrix(),
    
    #' @field S_Sigma_inv an \code{NxN} precision matrix of the second-level 
    #' Wishart prior distribution for the global parameter matrix \eqn{\mathbf{\Sigma}}.
    S_Sigma_inv = matrix(),
    
    #' @field eta a positive shape parameter of the second-level MNIW prior distribution
    #' for the global parameter matrices \eqn{\mathbf{A}}
    #' and \eqn{\mathbf{V}}
    eta = NA,
    
    #' @field mu_Sigma a positive shape parameter of the second-level Wishart prior 
    #' distribution  for the global parameter matrix \eqn{\mathbf{\Sigma}}.
    mu_Sigma  = NA,
    
    #' @field lambda a positive shape of the second-level exp prior distribution 
    #' for the shape parameter \eqn{\nu}.
    lambda  = NA,
    
    #' @field mu_m a scalar mean of the third-level normal prior distribution
    #' for the global average persistence parameter \eqn{m}.
    mu_m   = NA,
    
    #' @field sigma2_m a positive scalar variance of the third-level normal prior 
    #' distribution for the global average persistence parameter \eqn{m}.
    sigma2_m  = NA,
    
    #' @field s_w a positive scalar scale of the third-level gamma prior 
    #' distribution for parameter \eqn{w}.
    s_w  = NA,
    
    #' @field a_w a positive scalar shape of the third-level gamma prior 
    #' distribution for parameter \eqn{w}.
    a_w  = NA,
    
    #' @field s_s a positive scalar scale parameter of the third-level 
    #' inverted-gamma 2 prior distribution for parameter \eqn{s}.
    s_s  = NA,
    
    #' @field nu_s a positive scalar shape parameter of the third-level 
    #' inverted-gamma 2 prior distribution for parameter \eqn{s}.
    nu_s  = NA,
    
    #' @description
    #' Create a new prior specification PriorBVARPANEL.
    #' @param C a positive integer - the number of countries in the data.
    #' @param N a positive integer - the number of dependent variables in the model.
    #' @param p a positive integer - the autoregressive lag order of the SVAR model.
    #' @param d a positive integer - the number of \code{exogenous} variables in the model.
    #' @param stationary an \code{N} logical vector - its element set to 
    #' \code{FALSE} sets the prior mean for the autoregressive parameters of the 
    #' \code{N}th equation to the white noise process, otherwise to random walk.
    #' @return A new prior specification PriorBVARPANEL.
    #' @examples 
    #' # a prior for 2-country, 3-variable example with one lag and stationary data
    #' prior = specify_prior_bvarPANEL$new(C = 2, N = 3, p = 1)
    #' prior$M
    #' 
    initialize = function(C, N, p, d = 0, stationary = rep(FALSE, N)) {
      stopifnot("Argument C must be a positive integer number." = C > 0 & C %% 1 == 0)
      stopifnot("Argument N must be a positive integer number." = N > 0 & N %% 1 == 0)
      stopifnot("Argument p must be a positive integer number." = p > 0 & p %% 1 == 0)
      stopifnot("Argument d must be a non-negative integer number." = d >= 0 & d %% 1 == 0)
      stopifnot("Argument stationary must be a logical vector of length N." = length(stationary) == N & is.logical(stationary))
    
      K                 = N * p + 1 + d
      self$M            = t(cbind(diag(as.numeric(!stationary)), matrix(0, N, K - N)))
      self$W            = diag(c(kronecker((1:p)^2, rep(1, N) ), rep(10, 1 + d)))
      self$S_inv        = diag(N)
      self$S_Sigma_inv  = diag(N)
      self$eta          = N + 1
      self$mu_Sigma     = N + 1
      self$lambda       = 72
      self$mu_m         = 1
      self$sigma2_m     = 1
      self$s_w          = 1
      self$a_w          = 1
      self$s_s          = 1
      self$nu_s         = 3
    }, # END initialize
    
    #' @description
    #' Returns the elements of the prior specification PriorBSVAR as a \code{list}.
    #' 
    #' @examples 
    #' # a prior for 2-coutnry, 3-variable example with four lags
    #' prior = specify_prior_bvarPANEL$new(C = 2, N = 3, p = 4)
    #' prior$get_prior() # show the prior as list
    #' 
    get_prior = function(){
      list(
        M        = self$M,
        W        = self$W,
        S_inv    = self$S_inv,
        S_Sigma_inv = self$S_Sigma_inv,
        eta      = self$eta,
        mu_Sigma = self$mu_Sigma,
        lambda   = self$lambda,
        mu_m     = self$mu_m,
        sigma2_m = self$sigma2_m,
        s_w      = self$s_w,
        a_w      = self$a_w,
        s_s      = self$s_s,
        nu_s     = self$nu_s
      )
    } # END get_prior
  ) # END public
) # END specify_prior_bvarPANEL







#' R6 Class Representing StartingValuesBVARPANEL
#'
#' @description
#' The class StartingValuesBVARPANEL presents starting values for the Bayesian
#' hierarchical panel VAR model.
#' 
#' @examples 
#' # starting values for a Bayesian Panel VAR
#' sv = specify_starting_values_bvarPANEL$new(C = 2, N = 3, p = 1)
#' 
#' @export
specify_starting_values_bvarPANEL = R6::R6Class(
  "StartingValuesBVARPANEL",
  
  public = list(
    
    #' @field A_c an \code{KxNxC} array of starting values for the local parameter 
    #' \eqn{\mathbf{A}_c}. 
    A_c           = array(),
    
    #' @field Sigma_c an \code{NxNxC} array of starting values for the local
    #' parameter \eqn{\mathbf{\Sigma}_c}. 
    Sigma_c       = array(),
    
    #' @field A an \code{KxN} matrix of starting values for the global parameter 
    #' \eqn{\mathbf{A}}. 
    A             = matrix(),
    
    #' @field V an \code{KxK} matrix of starting values for the global parameter 
    #' \eqn{\mathbf{V}}. 
    V             = matrix(),
    
    #' @field Sigma an \code{NxN} matrix of starting values for the global parameter 
    #' \eqn{\mathbf{\Sigma}}. 
    Sigma         = matrix(),
    
    #' @field nu a positive scalar with starting values for the global parameter
    #' \eqn{\nu}.
    nu            = NA,
    
    #' @field m a positive scalar with starting values for the global hyper-parameter
    #' \eqn{m}.
    m             = NA,
    
    #' @field w a positive scalar with starting values for the global hyper-parameter
    #' \eqn{w}.
    w             = NA,
    
    #' @field s a positive scalar with starting values for the global hyper-parameter
    #' \eqn{s}.
    s             = NA,
    
    #' @description
    #' Create new starting values StartingValuesBVARPANEL
    #' @param C a positive integer - the number of countries in the data.
    #' @param N a positive integer - the number of dependent variables in the model.
    #' @param p a positive integer - the autoregressive lag order of the SVAR model.
    #' @param d a positive integer - the number of \code{exogenous} variables in the model.
    #' @return Starting values StartingValuesBVARPANEL
    #' @examples 
    #' # starting values for Bayesian Panel VAR 2-country model with 4 lags for a 3-variable system.
    #' sv = specify_starting_values_bvarPANEL$new(C = 2, N = 3, p = 4)
    #' 
    initialize = function(C, N, p, d = 0){
      stopifnot("Argument C must be a positive integer number." = C > 0 & C %% 1 == 0)
      stopifnot("Argument N must be a positive integer number." = N > 0 & N %% 1 == 0)
      stopifnot("Argument p must be a positive integer number." = p > 0 & p %% 1 == 0)
      stopifnot("Argument d must be a non-negative integer number." = d >= 0 & d %% 1 == 0)
      
      K               = N * p + 1 + d
      self$A_c        = array(stats::rnorm(C * K * N, sd = 0.001), c(K, N, C))
      self$Sigma_c    = stats::rWishart(C, N + 1, diag(N))
      self$A          = matrix(stats::rnorm(K * N, sd = 0.001), K, N) + diag(K)[,1:N]
      self$V          = stats::rWishart(1, K + 1, diag(K))[,,1]
      self$Sigma      = stats::rWishart(1, N + 1, diag(N))[,,1]
      self$nu         = N + 1 + 0.1
      self$m          = stats::rnorm(1, sd = 0.001)
      self$w          = stats::rgamma(1, 1)
      self$s          = stats::rgamma(1, 1)
    }, # END initialize
    
    #' @description
    #' Returns the elements of the starting values StartingValuesBVARPANEL as 
    #' a \code{list}.
    #' 
    #' @examples 
    #' # starting values for a homoskedastic bsvar with 1 lag for a 3-variable system
    #' sv = specify_starting_values_bvarPANEL$new(C = 2, N = 3, p = 1)
    #' sv$get_starting_values()   # show starting values as list
    #' 
    get_starting_values   = function(){
      list(
        A_c           = self$A_c,
        Sigma_c       = self$Sigma_c,
        A             = self$A,
        V             = self$V,
        Sigma         = self$Sigma,
        nu            = self$nu,
        m             = self$m,
        w             = self$w,
        s             = self$s
      )
    }, # END get_starting_values
    
    #' @description
    #' Returns the elements of the starting values StartingValuesBVARPANEL as a \code{list}.
    #' @param last_draw a list containing the same elements as object StartingValuesBVARPANEL.
    #' @return An object of class StartingValuesBVARPANEL including the last draw 
    #' of the current MCMC as the starting value to be passed to the continuation 
    #' of the MCMC estimation.
    #' 
    #' @examples
    #' sv = specify_starting_values_bvarPANEL$new(C = 2, N = 3, p = 1)
    #' 
    #' # Modify the starting values by:
    #' sv_list = sv$get_starting_values()   # getting them as list
    #' sv_list$A <- matrix(rnorm(12), 3, 4) # modifying the entry
    #' sv$set_starting_values(sv_list)      # providing to the class object
    #' 
    set_starting_values   = function(last_draw) {
      self$A_c            = last_draw$A_c
      self$Sigma_c        = last_draw$Sigma_c
      self$A              = last_draw$A
      self$V              = last_draw$V
      self$Sigma          = last_draw$Sigma
      self$nu             = last_draw$nu
      self$m              = last_draw$m
      self$w              = last_draw$w
      self$s              = last_draw$s
    } # END set_starting_values
  ) # END public
) # END specify_starting_values_bvarPANEL




#' R6 Class Representing DataMatricesBVARPANEL
#'
#' @description
#' The class DataMatricesBVARPANEL presents the data matrices of dependent 
#' variables, \eqn{\mathbf{Y}_c}, and regressors, \eqn{\mathbf{X}_c}, for the 
#' Bayesian Panel VAR model for all countries \eqn{c = 1, ..., C}.
#' 
#' @examples 
#' data(ilo_dynamic_panel)
#' YX = specify_panel_data_matrices$new(data = ilo_dynamic_panel, p = 4)
#' length(YX$Y); names(YX$Y)
#'
#' @export
specify_panel_data_matrices = R6::R6Class(
  "DataMatricesBVARPANEL",
  
  public = list(
    
    #' @field Y a list with \code{C} elements with \code{(T_c + p) x N} matrices 
    #' of dependent variables, \eqn{\mathbf{Y}_c}, possibly with missing 
    #' observations given by \code{NA}. 
    Y       = list(),
    
    #' @field missing a list with \code{C} elements with \code{T_c x N} matrices
    #' containing value \code{1} for missing observation and \code{0} otherwise.
    missing = list(),
    
    #' @field type an \code{N} character vector with elements set to "rate" or "real"
    #' determining the truncation of the predictive density to \code{[0, 100]} and
    #' \code{(-Inf, Inf)} (no truncation) for each of the variables.
    type  = character(),
    
    #' @field exogenous a list with \code{C} elements with \code{(T_c + p) x N} 
    #' matrices of exogenous variables. 
    exogenous = list(),
    
    #' @description
    #' Create new data matrices DataMatricesBVARPANEL
    #' @param data a list containing \code{(T_c+p)xN} matrices with country-specific
    #' time series data.
    #' @param p a positive integer providing model's autoregressive lag order.
    #' @param exogenous a list containing \code{(T_c+p)xd} matrices with 
    #' country-specific of exogenous variables. This matrix should not include a 
    #' constant term.
    #' @param type an \code{N} character vector with elements set to "rate" or "real"
    #' determining the truncation of the predictive density to \code{[0, 100]} and
    #' \code{(-Inf, Inf)} (no truncation) for each of the variables.
    #' @return New data matrices DataMatricesBVARPANEL
    initialize = function(data, p = 1L, exogenous = NULL, type = rep("real", ncol(data[[1]]))) {
      if (missing(data)) {
        stop("Argument data has to be specified")
      } else {
        stopifnot("Argument data has to be a list of matrices." = is.list(data) & all(simplify2array(lapply(data, function(x){is.matrix(x) & is.numeric(x)}))))
        stopifnot("Argument data has to contain matrices with the same number of columns." = length(unique(simplify2array(lapply(data, ncol)))) == 1)
      }
      stopifnot("Argument p must be a positive integer number." = p > 0 & p %% 1 == 0)
      
      stopifnot("Argument type must include elements 'rate' or 'real'." 
                = all(unique(type) %in% c("rate", "real")) )
      stopifnot("Argument type must be of length corresponding to the numbers of variables."
                = length(type) == ncol(data[[1]]) )
      
      C             = length(data)
      if (is.null(exogenous)) {
        d = 0
      } else {
        stopifnot("Argument exogenous has to be a list of matrices." = is.list(exogenous) & all(simplify2array(lapply(exogenous, function(x){is.matrix(x) & is.numeric(x)}))))
        stopifnot("Argument exogenous has to contain matrices with the same number of rows as argument data." = unique(simplify2array(lapply(exogenous, function(x){ncol(x)}))) >= 1 & unique(simplify2array(lapply(data, function(x){nrow(x)}))) == unique(simplify2array(lapply(exogenous, function(x){nrow(x)}))))
        stopifnot("Argument exogenous cannot include missing values." = unique(simplify2array(lapply(exogenous, function(x){any(is.na(x))}))) == FALSE )
        d = ncol(exogenous[[1]])
        Td = nrow(exogenous[[1]])
        test_exogenous = 0
        for (c in 1:C) {
          for (i in 1:d) {
            test_exogenous = test_exogenous + prod(exogenous[[c]][,i] - mean(exogenous[[c]][,i]) == rep(0,Td))
          }
        }
        stopifnot("Argument exogenous cannot include a constant term." = test_exogenous == 0 )
      }
      
      for (c in 1:C) {
        T             = nrow(data[[c]])
        y             = data[[c]]
        missing_c     = matrix(
          as.numeric(is.na(y)), 
          ncol = ncol(y)
        )
        my            = colMeans(y, na.rm = TRUE)
        y             = sapply(
          1:ncol(y), 
          function(i){
            y[is.na(y[,i]),i] <- my[i]
            y[,i]
          }
        )
        y = rbind(matrix(rep(my, p), nrow = p, byrow = TRUE), y)
        
        rownames(y)   = c(rep("", p), 1:T)
        colnames(y)   = paste0("v", 1:ncol(y))
        
        if ( any(class(data[[c]]) == "ts") ) {
          rownames(y) = c(rep(NA, p), as.numeric(time(data[[c]])))
        }  
        if ( !is.null(colnames(data[[c]])) ) {
          colnames(y) = colnames(data[[c]])
        }
        
        exo = as.matrix(rep(1, T + p))
        if ( !is.null(exogenous) ) {
          exo = cbind(
            exo, 
            rbind(
              matrix(0, nrow = p, ncol = d),
              exogenous[[c]]
            )
          )
        }
        
        self$Y[[c]]         = y
        self$missing[[c]]   = missing_c
        self$exogenous[[c]] = exo
        
      } # END c loop
      names(self$Y)         = names(data)
      names(self$missing)   = names(data)
      names(self$exogenous) = names(data)
      self$type             = type
    }, # END initialize
    
    #' @description
    #' Returns the data matrices DataMatricesBVARPANEL as a \code{list}.
    #' 
    #' @examples 
    #' data(ilo_dynamic_panel)
    #' YX = specify_panel_data_matrices$new(ilo_dynamic_panel)
    #' YX$get_data_matrices()
    #' 
    get_data_matrices = function() {
      list(
        Y       = self$Y,
        missing = self$missing,
        exogenous = self$exogenous,
        type    = self$type
      )
    } # END get_data_matrices
  ) # END public
) # END specify_panel_data_matrices







#' R6 Class representing the specification of the BVARPANEL model
#'
#' @description
#' The class BVARPANEL presents complete specification for the Bayesian Panel
#' Vector Autoregression.
#' 
#' @examples 
#' spec = specify_bvarPANEL$new(
#'    data = ilo_dynamic_panel
#' )
#' 
#' @export
specify_bvarPANEL = R6::R6Class(
  "BVARPANEL",
  
  private = list(
    type  = "wozniak" # a value from set, \code{wozniak}, \code{jarocinski}, indicating model specification
  ),
  
  public = list(
    
    #' @field p a non-negative integer specifying the autoregressive lag order of the model. 
    p                      = numeric(),
    
    #' @field prior an object PriorBSVAR with the prior specification. 
    prior                  = list(),
    
    #' @field data_matrices an object DataMatricesBVARPANEL with the data matrices.
    data_matrices          = list(),
    
    #' @field starting_values an object StartingValuesBVARPANEL with the starting values.
    starting_values        = list(),
    
    #' @field adaptiveMH a vector of four values setting the adaptive MH sampler 
    #' for nu: adaptive rate, target acceptance rate, the iteration at which to 
    #' start adapting, the initial scaling rate
    adaptiveMH             = numeric(),
    
    #' @description
    #' Create a new specification of the Bayesian Panel VAR model BVARPANEL.
    #' @param data a list with \code{C} elements of \code{(T_c+p)xN} matrices 
    #' with time series data.
    #' @param p a positive integer providing model's autoregressive lag order.
    #' @param exogenous a \code{(T+p)xd} matrix of exogenous variables. 
    #' @param stationary an \code{N} logical vector - its element set to 
    #' \code{FALSE} sets the prior mean for the autoregressive parameters of the 
    #' \code{N}th equation to the white noise process, otherwise to random walk.
    #' @param type an \code{N} character vector with elements set to "rate" or "real"
    #' determining the truncation of the predictive density to \code{[0, 100]} and
    #' \code{(-Inf, Inf)} (no truncation) for each of the variables.
    #' @return A new complete specification for the Bayesian Panel VAR model BVARPANEL.
    initialize = function(
    data,
    p = 1L,
    exogenous = NULL,
    stationary = rep(FALSE, ncol(data[[1]])),
    type = rep("real", ncol(data[[1]]))
    ) {
      stopifnot("Argument data has to contain matrices with the same number of columns." 
                = length(unique(simplify2array(lapply(data, ncol)))) == 1)
      stopifnot("Argument p has to be a positive integer." 
                = ((p %% 1) == 0 & p > 0))
      
      self$p    = p
      C         = length(data)
      N         = unique(simplify2array(lapply(data, ncol)))
      d             = 0
      if (!is.null(exogenous)) {
        d           = ncol(exogenous[[1]])
      }
    
      self$data_matrices   = specify_panel_data_matrices$new(data, self$p, exogenous, type)
      self$prior           = specify_prior_bvarPANEL$new(C, N, self$p, d, stationary)
      self$starting_values = specify_starting_values_bvarPANEL$new(C, N, self$p, d)
      self$adaptiveMH      = c(0.44, 0.6)
      
    }, # END initialize

    #' @description
    #' Sets the model in line with the specification by Jarocinski (2010) as presented 
    #' by Dieppe, Legrand, Roye (2016).
    #' 
    #' @references
    #' Jarocinski (2010). Responses to monetary policy shocks in the east and 
    #' the west of Europe: a comparison. \emph{Journal of Applied Econometrics}, 
    #' \bold{25}(5), 833-868, \doi{10.1002/jae.1082}.
    #' 
    #' Dieppe, Legrand, Roye (2016). The BEAR toolbox, \emph{ECB Working Papers},
    #' \bold{1934}, \doi{10.2866/292952}.
    #' 
    #' @examples
    #' spec = specify_bvarPANEL$new(
    #'    data = ilo_dynamic_panel
    #' )
    #' spec$set_to_Jarocinski()
    #' 
    set_to_Jarocinski = function() {
      
      message("Setting the model in line with the specification by Jarocinski (2010).")
      private$type                = "jarocinski"
      
      N                           = nrow(self$starting_values$Sigma)
      K                           = nrow(self$prior$M)
      
      # values used in computations
      self$starting_values$V      = self$prior$W  # self$starting_values$s * self$prior$W
      self$starting_values$s      = 1             # A_c prior overall shrinkage
      self$prior$nu_s             = 0.001         # s prior shape
      self$prior$s_s              = 0.001         # s prior scale
      self$starting_values$nu     = -(N + 1)      # nu value imposing Jarocinski prior
      
      
      # values NOT used
      self$starting_values$Sigma  = matrix(0, N, N)
      self$starting_values$m      = 0
      self$starting_values$w      = 1
      
      self$prior$M                = matrix(0, K, N)
      self$prior$S_inv            = matrix(0, N, N)
      self$prior$S_Sigma_inv      = matrix(0, N, N)
      self$prior$eta              = 0
      self$prior$mu_Sigma         = 0
      self$prior$lambda           = 0
      self$prior$mu_m             = 0
      self$prior$sigma2_m         = 0
      self$prior$s_w              = 0
      self$prior$a_w              = 0
      
    }, # END set_prior_Jarocinski
    
    #' @description
    #' Returns the data matrices as the DataMatricesBVARPANEL object.
    #' 
    #' @examples
    #' spec = specify_bvarPANEL$new(
    #'    data = ilo_dynamic_panel
    #' )
    #' spec$get_data_matrices()
    #' 
    get_data_matrices = function() {
      self$data_matrices$clone()
    }, # END get_data_matrices
    
    #' @description
    #' Returns the prior specification as the PriorBVARPANEL object.
    #' 
    #' @examples 
    #' spec = specify_bvarPANEL$new(
    #'    data = ilo_dynamic_panel
    #' )
    #' spec$get_prior()
    #' 
    get_prior = function() {
      self$prior$clone()
    }, # END get_prior
    
    #' @description
    #' Returns the starting values as the StartingValuesBVARPANEL object.
    #' 
    #' @examples 
    #' spec = specify_bvarPANEL$new(
    #'    data = ilo_dynamic_panel
    #' )
    #' spec$get_starting_values()
    #' 
    get_starting_values = function() {
      self$starting_values$clone()
    }, # END get_starting_values
    
    #' @description
    #' Returns the type of the model.
    #' 
    #' @examples 
    #' spec = specify_bvarPANEL$new(
    #'    data = ilo_dynamic_panel
    #' )
    #' spec$get_type()
    #' 
    get_type = function() {
      private$type
    }, # END get_starting_values
    
    #' @description
    #' Sets the prior mean of the global autoregressive parameters to the OLS 
    #' pooled panel estimator following Zellner, Hong (1989).
    #' 
    #' @param x a vector of four values setting the adaptive MH sampler for nu:
    #' adaptive rate, target acceptance rate, the iteration at which to 
    #' start adapting, the initial scaling rate
    #' 
    #' @references
    #' Zellner, Hong (1989). Forecasting international growth rates using 
    #' Bayesian shrinkage and other procedures. \emph{Journal of Econometrics}, 
    #' \bold{40}(1), 183–202, \doi{10.1016/0304-4076(89)90036-5}.
    #' 
    #' @examples 
    #' spec = specify_bvarPANEL$new(
    #'    data = ilo_dynamic_panel
    #' )
    #' spec$set_global2pooled()
    #' 
    set_global2pooled = function(x) {
      stopifnot("This model does feature such a specification." 
                = private$type == "wozniak")
      
      C = length(self$data_matrices$Y)
      N = ncol(self$data_matrices$Y[[1]])
      d = ncol(self$data_matrices$exogenous[[1]])
      p = self$p
      K = N * p + d
      
      XX = matrix(0, K, K)
      XY = matrix(0, K, N)
      for (c in 1:C) {
        XcYc_tmp = .Call(`_bpvars_Y_c_and_X_c`, 
                         self$data_matrices$Y[[c]], 
                         self$data_matrices$exogenous[[c]],
                         p)
        
        Yc = XcYc_tmp[1,][[1]]
        Xc = XcYc_tmp[2,][[1]]
        
        not_missing = !apply(self$data_matrices$missing[[c]], 1, \(x)(any(x == 1)))
        if (sum(not_missing) != 0) {
          Yc = Yc[not_missing,]
          Xc = Xc[not_missing,]
          XX = XX + crossprod(Xc)
          XY = XY + crossprod(Xc, Yc)
        }
      }
      self$prior$M = solve(XX, XY)
    }, # END set_adaptiveMH
    
    #' @description
    #' Sets the parameters of adaptive Metropolis-Hastings sampler for the parameter nu.
    #' 
    #' @param x a vector of four values setting the adaptive MH sampler for nu:
    #' adaptive rate, target acceptance rate, the iteration at which to 
    #' start adapting, the initial scaling rate
    #' 
    #' @examples 
    #' spec = specify_bvarPANEL$new(
    #'    data = ilo_dynamic_panel
    #' )
    #' spec$set_adaptiveMH(c(0.6, 0.4, 10, 0.1))
    #' 
    set_adaptiveMH = function(x) {
      stopifnot("Argument x has to be a numeric vector of length 4." = length(x) == 4 & is.numeric(x))
      stopifnot("Argument x must contain positive values." = all(x > 0))
      stopifnot("The second element of argument x must be less than 1." = x[2] < 1)
      stopifnot("The third element of argument x must greater or equal to 1." = x[3] >= 1)
      x[3]            = floor(x[3])
      self$adaptiveMH = x
    } # END set_adaptiveMH
  ) # END public
) # END specify_bvarPANEL




#' R6 Class Representing PosteriorBVARPANEL
#'
#' @description
#' The class PosteriorBVARPANEL contains posterior output and the specification 
#' including the last MCMC draw for the Bayesian Panel VAR model. 
#' Note that due to the thinning of the MCMC output the starting value in element 
#' \code{last_draw} might not be equal to the last draw provided in 
#' element \code{posterior}.
#' 
#' @seealso \code{\link{specify_bvarPANEL}}
#' 
#' @examples 
#' specification = specify_bvarPANEL$new(
#'    data = ilo_dynamic_panel[1:5]
#' )
#' posterior       = estimate(specification, 5)
#' class(posterior)
#' 
#' @export
specify_posterior_bvarPANEL = R6::R6Class(
  "PosteriorBVARPANEL",
  
  private = list(
    normalised = FALSE
  ), # END private
  
  public = list(
    
    #' @field last_draw an object of class BVARPANEL with the last draw of the 
    #' current MCMC run as the starting value to be passed to the continuation 
    #' of the MCMC estimation using \code{estimate()}. 
    last_draw = list(),
    
    #' @field posterior a list containing Bayesian estimation output.
    posterior = list(),
    
    #' @description
    #' Create a new posterior output PosteriorBVARPANEL.
    #' @param specification_bvarPANEL an object of class BVARPANEL with the last 
    #' draw of the current MCMC run as the starting value.
    #' @param posterior_bvarPANEL a list containing Bayesian estimation output.
    #' @return A posterior output PosteriorBVARPANEL.
    initialize = function(specification_bvarPANEL, posterior_bvarPANEL) {
      
      stopifnot("Argument specification_bvarPANEL must be of class BVARPANEL." = any(class(specification_bvarPANEL) == "BVARPANEL"))
      stopifnot("Argument posterior_bvarPANEL must must contain MCMC output." = is.list(posterior_bvarPANEL) & is.array(posterior_bvarPANEL$A) & is.array(posterior_bvarPANEL$Sigma) & is.array(posterior_bvarPANEL$V))
      
      self$last_draw    = specification_bvarPANEL
      self$posterior    = posterior_bvarPANEL
      
      N = dim(specification_bvarPANEL$starting_values$A_c)[2]
      K = dim(specification_bvarPANEL$starting_values$A_c)[1]
      C = dim(specification_bvarPANEL$starting_values$A_c)[3]
      S = dim(posterior_bvarPANEL$A)[3]
      
      Sigma_c           = array(NA, c(N, N, C, S))
      A_c               = array(NA, c(K, N, C, S))
      for (s in 1:S) {
        A_c[,,,s]       = posterior_bvarPANEL$A_c_cpp[s,1][[1]]
        Sigma_c[,,,s]   = posterior_bvarPANEL$Sigma_c_cpp[s,1][[1]]
      }
      self$posterior$Sigma_c   = Sigma_c
      self$posterior$A_c       = A_c
      
      
    }, # END initialize
    
    #' @description
    #' Returns a list containing Bayesian estimation output.
    #' 
    #' @examples 
    #' specification = specify_bvarPANEL$new(
    #'    data = ilo_dynamic_panel[1:5]
    #' )
    #' posterior       = estimate(specification, 5)
    #' posterior$get_posterior()
    #' 
    get_posterior       = function(){
      self$posterior
    }, # END get_posterior
    
    #' @description
    #' Returns an object of class BVARPANEL with the last draw of the current 
    #' MCMC run as the starting value to be passed to the continuation of the 
    #' MCMC estimation using \code{estimate()}.
    #' 
    #' @examples
    #' specification = specify_bvarPANEL$new(
    #'    data = ilo_dynamic_panel[1:5]
    #' )
    #' burn_in        = estimate(specification, 5)
    #' posterior      = estimate(burn_in, 5)
    #' 
    get_last_draw      = function(){
      self$last_draw$clone()
    } # END get_last_draw
    
  ) # END public
) # END specify_posterior_bvarPANEL


