/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Omodule.h" 
#define H5T_FRIEND     

#include "H5private.h"   
#include "H5Dprivate.h"  
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5FLprivate.h" 
#include "H5MMprivate.h" 
#include "H5Opkg.h"      
#include "H5Tpkg.h"      
#include "H5VMprivate.h" 

static herr_t H5O__dtype_encode(H5F_t *f, uint8_t *p, const void *mesg);
static void  *H5O__dtype_decode(H5F_t *f, H5O_t *open_oh, unsigned mesg_flags, unsigned *ioflags,
                                size_t p_size, const uint8_t *p);
static void  *H5O__dtype_copy(const void *_mesg, void *_dest);
static size_t H5O__dtype_size(const H5F_t *f, const void *_mesg);
static herr_t H5O__dtype_reset(void *_mesg);
static herr_t H5O__dtype_free(void *_mesg);
static herr_t H5O__dtype_set_share(void *_mesg, const H5O_shared_t *sh);
static htri_t H5O__dtype_can_share(const void *_mesg);
static herr_t H5O__dtype_pre_copy_file(H5F_t *file_src, const void *mesg_src, bool *deleted,
                                       const H5O_copy_t *cpy_info, void *_udata);
static void  *H5O__dtype_copy_file(H5F_t *file_src, const H5O_msg_class_t *mesg_type, void *native_src,
                                   H5F_t *file_dst, bool *recompute_size, H5O_copy_t *cpy_info, void *udata);
static herr_t H5O__dtype_shared_post_copy_upd(const H5O_loc_t *src_oloc, const void *mesg_src,
                                              H5O_loc_t *dst_oloc, void *mesg_dst, H5O_copy_t *cpy_info);
static herr_t H5O__dtype_debug(H5F_t *f, const void *_mesg, FILE *stream, int indent, int fwidth);

#define H5O_SHARED_TYPE        H5O_MSG_DTYPE
#define H5O_SHARED_DECODE      H5O__dtype_shared_decode
#define H5O_SHARED_DECODE_REAL H5O__dtype_decode
#define H5O_SHARED_ENCODE      H5O__dtype_shared_encode
#define H5O_SHARED_ENCODE_REAL H5O__dtype_encode
#define H5O_SHARED_SIZE        H5O__dtype_shared_size
#define H5O_SHARED_SIZE_REAL   H5O__dtype_size
#define H5O_SHARED_DELETE      H5O__dtype_shared_delete
#undef H5O_SHARED_DELETE_REAL
#define H5O_SHARED_LINK H5O__dtype_shared_link
#undef H5O_SHARED_LINK_REAL
#define H5O_SHARED_COPY_FILE      H5O__dtype_shared_copy_file
#define H5O_SHARED_COPY_FILE_REAL H5O__dtype_copy_file
#define H5O_SHARED_POST_COPY_FILE H5O__dtype_shared_post_copy_file
#undef H5O_SHARED_POST_COPY_FILE_REAL
#define H5O_SHARED_POST_COPY_FILE_UPD H5O__dtype_shared_post_copy_upd
#define H5O_SHARED_DEBUG              H5O__dtype_shared_debug
#define H5O_SHARED_DEBUG_REAL         H5O__dtype_debug
#include "H5Oshared.h" 

#ifdef H5_STRICT_FORMAT_CHECKS

#define H5O_DTYPE_CHECK_VERSION(DT, VERS, MIN_VERS, IOF, CLASS, ERR)                                         \
    if (((VERS) < (MIN_VERS)) && !(*(IOF) & H5O_DECODEIO_NOCHANGE))                                          \
        HGOTO_ERROR(H5E_DATATYPE, H5E_VERSION, ERR, "incorrect " CLASS " datatype version");
#else 

#define H5O_DTYPE_CHECK_VERSION(DT, VERS, MIN_VERS, IOF, CLASS, ERR)                                         \
    if (((VERS) < (MIN_VERS)) && !(*(IOF) & H5O_DECODEIO_NOCHANGE)) {                                        \
        (VERS) = (MIN_VERS);                                                                                 \
        if (H5T__upgrade_version((DT), (VERS)) < 0)                                                          \
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTSET, FAIL, "can't upgrade " CLASS " encoding version");        \
        *(IOF) |= H5O_DECODEIO_DIRTY;                                                                        \
    }  
#endif 

const H5O_msg_class_t H5O_MSG_DTYPE[1] = {{
    H5O_DTYPE_ID,                              
    "datatype",                                
    sizeof(H5T_t),                             
    H5O_SHARE_IS_SHARABLE | H5O_SHARE_IN_OHDR, 
    H5O__dtype_shared_decode,                  
    H5O__dtype_shared_encode,                  
    H5O__dtype_copy,                           
    H5O__dtype_shared_size,                    
    H5O__dtype_reset,                          
    H5O__dtype_free,                           
    H5O__dtype_shared_delete,                  
    H5O__dtype_shared_link,                    
    H5O__dtype_set_share,                      
    H5O__dtype_can_share,                      
    H5O__dtype_pre_copy_file,                  
    H5O__dtype_shared_copy_file,               
    H5O__dtype_shared_post_copy_file,          
    NULL,                                      
    NULL,                                      
    H5O__dtype_shared_debug                    
}};

static htri_t
H5O__dtype_decode_helper(unsigned *ioflags , const uint8_t **pp, H5T_t *dt, bool skip,
                         const uint8_t *p_end)
{
    unsigned flags;
    unsigned version;
    htri_t   ret_value = false;

    FUNC_ENTER_PACKAGE

    assert(pp && *pp);
    assert(dt && dt->shared);

    

    
    if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, 4, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
    UINT32DECODE(*pp, flags);
    version = (flags >> 4) & 0x0f;
    if (version < H5O_DTYPE_VERSION_1 || version > H5O_DTYPE_VERSION_LATEST)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTLOAD, FAIL, "bad version number for datatype message");
    dt->shared->version = version;
    dt->shared->type    = (H5T_class_t)(flags & 0x0f);
    flags >>= 8;

    
    if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, 4, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
    UINT32DECODE(*pp, dt->shared->size);

    
    if (dt->shared->size == 0)
        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL, "invalid datatype size");

    switch (dt->shared->type) {
        case H5T_INTEGER:
            
            dt->shared->u.atomic.order    = (flags & 0x1) ? H5T_ORDER_BE : H5T_ORDER_LE;
            dt->shared->u.atomic.lsb_pad  = (flags & 0x2) ? H5T_PAD_ONE : H5T_PAD_ZERO;
            dt->shared->u.atomic.msb_pad  = (flags & 0x4) ? H5T_PAD_ONE : H5T_PAD_ZERO;
            dt->shared->u.atomic.u.i.sign = (flags & 0x8) ? H5T_SGN_2 : H5T_SGN_NONE;
            if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, 2 + 2, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
            UINT16DECODE(*pp, dt->shared->u.atomic.offset);
            UINT16DECODE(*pp, dt->shared->u.atomic.prec);

            
            if (dt->shared->u.atomic.offset >= (dt->shared->size * 8))
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADRANGE, FAIL, "integer offset out of bounds");
            if (0 == dt->shared->u.atomic.prec)
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADVALUE, FAIL, "precision is zero");
            if (((dt->shared->u.atomic.offset + dt->shared->u.atomic.prec) - 1) >= (dt->shared->size * 8))
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADRANGE, FAIL, "integer offset+precision out of bounds");
            break;

        case H5T_FLOAT:
            
            dt->shared->u.atomic.order = (flags & 0x1) ? H5T_ORDER_BE : H5T_ORDER_LE;
            if (version >= H5O_DTYPE_VERSION_3) {
                
                if ((flags & 0x40) && !(flags & 0x1))
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "bad byte order for datatype message");

                
                if (flags & 0x40)
                    dt->shared->u.atomic.order = H5T_ORDER_VAX;
            }
            dt->shared->u.atomic.lsb_pad = (flags & 0x2) ? H5T_PAD_ONE : H5T_PAD_ZERO;
            dt->shared->u.atomic.msb_pad = (flags & 0x4) ? H5T_PAD_ONE : H5T_PAD_ZERO;
            dt->shared->u.atomic.u.f.pad = (flags & 0x8) ? H5T_PAD_ONE : H5T_PAD_ZERO;
            switch ((flags >> 4) & 0x03) {
                case 0:
                    dt->shared->u.atomic.u.f.norm = H5T_NORM_NONE;
                    break;

                case 1:
                    dt->shared->u.atomic.u.f.norm = H5T_NORM_MSBSET;
                    break;

                case 2:
                    dt->shared->u.atomic.u.f.norm = H5T_NORM_IMPLIED;
                    break;

                default:
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "unknown floating-point normalization");
            }
            dt->shared->u.atomic.u.f.sign = (flags >> 8) & 0xff;
            if (dt->shared->u.atomic.u.f.sign >= (dt->shared->size * 8))
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADRANGE, FAIL, "sign bit position out of bounds");

            if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, 2 + 2, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
            UINT16DECODE(*pp, dt->shared->u.atomic.offset);
            UINT16DECODE(*pp, dt->shared->u.atomic.prec);

            if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, 1 + 1, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
            dt->shared->u.atomic.u.f.epos  = *(*pp)++;
            dt->shared->u.atomic.u.f.esize = *(*pp)++;
            if (dt->shared->u.atomic.u.f.esize == 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADVALUE, FAIL, "exponent size can't be zero");
            if (dt->shared->u.atomic.u.f.epos >= (dt->shared->size * 8))
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADRANGE, FAIL, "exponent starting position out of bounds");
            if (((dt->shared->u.atomic.u.f.epos + dt->shared->u.atomic.u.f.esize) - 1) >=
                (dt->shared->size * 8))
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADRANGE, FAIL, "exponent range out of bounds");

            if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, 1 + 1, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
            dt->shared->u.atomic.u.f.mpos  = *(*pp)++;
            dt->shared->u.atomic.u.f.msize = *(*pp)++;
            if (dt->shared->u.atomic.u.f.msize == 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADVALUE, FAIL, "mantissa size can't be zero");
            if (dt->shared->u.atomic.u.f.mpos >= (dt->shared->size * 8))
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADRANGE, FAIL, "mantissa starting position out of bounds");
            if (((dt->shared->u.atomic.u.f.mpos + dt->shared->u.atomic.u.f.msize) - 1) >=
                (dt->shared->size * 8))
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADRANGE, FAIL, "mantissa range out of bounds");

            if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, 4, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
            UINT32DECODE(*pp, dt->shared->u.atomic.u.f.ebias);

            
            if (H5_RANGE_OVERLAP(dt->shared->u.atomic.u.f.sign, dt->shared->u.atomic.u.f.sign,
                                 dt->shared->u.atomic.u.f.epos,
                                 ((dt->shared->u.atomic.u.f.epos + dt->shared->u.atomic.u.f.esize) - 1)))
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADVALUE, FAIL, "exponent and sign positions overlap");
            if (H5_RANGE_OVERLAP(dt->shared->u.atomic.u.f.sign, dt->shared->u.atomic.u.f.sign,
                                 dt->shared->u.atomic.u.f.mpos,
                                 ((dt->shared->u.atomic.u.f.mpos + dt->shared->u.atomic.u.f.msize) - 1)))
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADVALUE, FAIL, "mantissa and sign positions overlap");
            if (H5_RANGE_OVERLAP(dt->shared->u.atomic.u.f.epos,
                                 ((dt->shared->u.atomic.u.f.epos + dt->shared->u.atomic.u.f.esize) - 1),
                                 dt->shared->u.atomic.u.f.mpos,
                                 ((dt->shared->u.atomic.u.f.mpos + dt->shared->u.atomic.u.f.msize) - 1)))
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADVALUE, FAIL, "mantissa and exponent positions overlap");
            break;

        case H5T_TIME:
            
            dt->shared->u.atomic.order = (flags & 0x1) ? H5T_ORDER_BE : H5T_ORDER_LE;
            if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, 2, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
            UINT16DECODE(*pp, dt->shared->u.atomic.prec);
            break;

        case H5T_STRING:
            
            dt->shared->u.atomic.order   = H5T_ORDER_NONE;
            dt->shared->u.atomic.prec    = 8 * dt->shared->size;
            dt->shared->u.atomic.offset  = 0;
            dt->shared->u.atomic.lsb_pad = H5T_PAD_ZERO;
            dt->shared->u.atomic.msb_pad = H5T_PAD_ZERO;

            dt->shared->u.atomic.u.s.pad  = (H5T_str_t)(flags & 0x0f);
            dt->shared->u.atomic.u.s.cset = (H5T_cset_t)((flags >> 4) & 0x0f);
            break;

        case H5T_BITFIELD:
            
            dt->shared->u.atomic.order   = (flags & 0x1) ? H5T_ORDER_BE : H5T_ORDER_LE;
            dt->shared->u.atomic.lsb_pad = (flags & 0x2) ? H5T_PAD_ONE : H5T_PAD_ZERO;
            dt->shared->u.atomic.msb_pad = (flags & 0x4) ? H5T_PAD_ONE : H5T_PAD_ZERO;
            if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, 2 + 2, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
            UINT16DECODE(*pp, dt->shared->u.atomic.offset);
            UINT16DECODE(*pp, dt->shared->u.atomic.prec);
            break;

        case H5T_OPAQUE: {
            size_t z;

            

            
            z = flags & (H5T_OPAQUE_TAG_MAX - 1);
            if (0 != (z & 0x7))
                HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL, "opaque flag field must be aligned");

            if (NULL == (dt->shared->u.opaque.tag = (char *)H5MM_malloc(z + 1)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

            if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, z, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
            H5MM_memcpy(dt->shared->u.opaque.tag, *pp, z);
            dt->shared->u.opaque.tag[z] = '\0';

            *pp += z;
            break;
        }

        case H5T_COMPOUND: {
            unsigned nmembs;           
            unsigned offset_nbytes;    
            size_t   max_memb_pos = 0; 
            unsigned max_version  = 0; 
            unsigned upgrade_to   = 0; 

            
            offset_nbytes = H5VM_limit_enc_size((uint64_t)dt->shared->size);

            
            nmembs = flags & 0xffff;
            if (nmembs == 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADVALUE, FAIL, "invalid number of members: %u", nmembs);
            if (NULL ==
                (dt->shared->u.compnd.memb = (H5T_cmemb_t *)H5MM_calloc(nmembs * sizeof(H5T_cmemb_t))))
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, FAIL, "memory allocation failed");
            dt->shared->u.compnd.nalloc = nmembs;

            if (dt->shared->u.compnd.memb_size != 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADVALUE, FAIL, "member size not initialized to zero");

            for (dt->shared->u.compnd.nmembs = 0; dt->shared->u.compnd.nmembs < nmembs;
                 dt->shared->u.compnd.nmembs++) {

                size_t   actual_name_length = 0; 
                unsigned ndims              = 0; 
                htri_t   can_upgrade;            
                hsize_t  dim[H5O_LAYOUT_NDIMS];  
                H5T_t   *array_dt;               
                H5T_t   *temp_type;              
                unsigned memb_idx;               

                
                if (!skip) {
                    

                    size_t max = (size_t)(p_end - *pp + 1); 

                    actual_name_length = strnlen((const char *)*pp, max);
                    if (actual_name_length == max)
                        HGOTO_ERROR(H5E_OHDR, H5E_NOSPACE, FAIL, "field name not null terminated");
                }
                else {
                    
                    actual_name_length = strlen((const char *)*pp);
                }

                if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, actual_name_length, p_end))
                    HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");

                
                for (memb_idx = 0; memb_idx < dt->shared->u.compnd.nmembs; memb_idx++)
                    if (0 == strcmp((const char *)*pp, dt->shared->u.compnd.memb[memb_idx].name))
                        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL,
                                    "duplicated compound field name '%s', for fields %u and %u",
                                    (const char *)*pp, memb_idx, dt->shared->u.compnd.nmembs);

                
                if (NULL == (dt->shared->u.compnd.memb[dt->shared->u.compnd.nmembs].name =
                                 H5MM_xstrdup((const char *)*pp)))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_CANTCOPY, FAIL,
                                "can't duplicate compound member name string");

                
                if (version >= H5O_DTYPE_VERSION_3) {
                    
                    if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, actual_name_length + 1, p_end))
                        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL,
                                    "ran off end of input buffer while decoding");
                    *pp += actual_name_length + 1;
                }
                else {
                    
                    if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, ((actual_name_length + 8) / 8) * 8, p_end))
                        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL,
                                    "ran off end of input buffer while decoding");
                    *pp += ((actual_name_length + 8) / 8) * 8;
                }

                
                
                if (version >= H5O_DTYPE_VERSION_3) {
                    H5_WARN_USELESS_COMPARISON_OFF
                    if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, offset_nbytes, p_end))
                        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL,
                                    "ran off end of input buffer while decoding");
                    H5_WARN_USELESS_COMPARISON_ON
                    UINT32DECODE_VAR(*pp, dt->shared->u.compnd.memb[dt->shared->u.compnd.nmembs].offset,
                                     offset_nbytes);
                }
                else {
                    if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, 4, p_end))
                        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL,
                                    "ran off end of input buffer while decoding");
                    UINT32DECODE(*pp, dt->shared->u.compnd.memb[dt->shared->u.compnd.nmembs].offset);
                }

                
                if (version == H5O_DTYPE_VERSION_1) {
                    
                    if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, 1, p_end))
                        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL,
                                    "ran off end of input buffer while decoding");
                    ndims = *(*pp)++;

                    
                    if (ndims > 4) {
                        dt->shared->u.compnd.memb[dt->shared->u.compnd.nmembs].name =
                            H5MM_xfree(dt->shared->u.compnd.memb[dt->shared->u.compnd.nmembs].name);
                        HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, FAIL,
                                    "invalid number of dimensions for array");
                    }

                    
                    if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, 3, p_end))
                        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL,
                                    "ran off end of input buffer while decoding");
                    *pp += 3;

                    
                    if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, 4, p_end))
                        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL,
                                    "ran off end of input buffer while decoding");
                    *pp += 4;

                    
                    if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, 4, p_end))
                        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL,
                                    "ran off end of input buffer while decoding");
                    *pp += 4;

                    
                    if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, (4 * 4), p_end))
                        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL,
                                    "ran off end of input buffer while decoding");
                    for (int i = 0; i < 4; i++)
                        UINT32DECODE(*pp, dim[i]);
                }

                
                if (NULL == (temp_type = H5T__alloc())) {
                    dt->shared->u.compnd.memb[dt->shared->u.compnd.nmembs].name =
                        H5MM_xfree(dt->shared->u.compnd.memb[dt->shared->u.compnd.nmembs].name);
                    HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");
                }

                
                if ((can_upgrade = H5O__dtype_decode_helper(ioflags, pp, temp_type, skip, p_end)) < 0) {
                    dt->shared->u.compnd.memb[dt->shared->u.compnd.nmembs].name =
                        H5MM_xfree(dt->shared->u.compnd.memb[dt->shared->u.compnd.nmembs].name);
                    if (H5T_close_real(temp_type) < 0)
                        HDONE_ERROR(H5E_DATATYPE, H5E_CANTRELEASE, FAIL, "can't release datatype info");
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTDECODE, FAIL, "unable to decode member type");
                }
                if (temp_type->shared->size == 0)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTDECODE, FAIL, "type size can't be zero");
                if ((dt->shared->u.compnd.memb[dt->shared->u.compnd.nmembs].offset +
                     temp_type->shared->size) > dt->shared->size) {
                    if (H5T_close_real(temp_type) < 0)
                        HDONE_ERROR(H5E_DATATYPE, H5E_CANTRELEASE, FAIL, "can't release datatype info");
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTDECODE, FAIL,
                                "member type extends outside its parent compound type");
                }

                
                if (can_upgrade && temp_type->shared->version > version) {
                    upgrade_to = temp_type->shared->version;

                    
                    ret_value = true;
                }

                
                if (version == H5O_DTYPE_VERSION_1) {
                    
                    if (ndims > 0) {
                        
                        for (unsigned u = 0; u < ndims; u++)
                            if (!(dim[u] > 0)) {
                                dt->shared->u.compnd.memb[dt->shared->u.compnd.nmembs].name =
                                    H5MM_xfree(dt->shared->u.compnd.memb[dt->shared->u.compnd.nmembs].name);
                                if (H5T_close_real(temp_type) < 0)
                                    HDONE_ERROR(H5E_DATATYPE, H5E_CANTRELEASE, FAIL,
                                                "can't release datatype info");
                                HGOTO_ERROR(H5E_DATATYPE, H5E_BADVALUE, FAIL,
                                            "zero-sized dimension specified");
                            }

                        
                        if ((array_dt = H5T__array_create(temp_type, ndims, dim)) == NULL) {
                            dt->shared->u.compnd.memb[dt->shared->u.compnd.nmembs].name =
                                H5MM_xfree(dt->shared->u.compnd.memb[dt->shared->u.compnd.nmembs].name);
                            if (H5T_close_real(temp_type) < 0)
                                HDONE_ERROR(H5E_DATATYPE, H5E_CANTRELEASE, FAIL,
                                            "can't release datatype info");
                            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, FAIL,
                                        "unable to create array datatype");
                        }

                        
                        if (H5T_close_real(temp_type) < 0) {
                            dt->shared->u.compnd.memb[dt->shared->u.compnd.nmembs].name =
                                H5MM_xfree(dt->shared->u.compnd.memb[dt->shared->u.compnd.nmembs].name);
                            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTRELEASE, FAIL, "can't release datatype info");
                        }

                        
                        temp_type = array_dt;

                        
                        if (*ioflags & H5O_DECODEIO_NOCHANGE)
                            temp_type->shared->version = H5O_DTYPE_VERSION_1;
                        else {
                            
                            if (upgrade_to < temp_type->shared->version)
                                upgrade_to = temp_type->shared->version;

                            
                            ret_value = true;
                        }
                    }
                }

                
                if (temp_type->shared->version > max_version)
                    max_version = temp_type->shared->version;

                
                if (temp_type->shared->force_conv == true)
                    dt->shared->force_conv = true;

                
                dt->shared->u.compnd.memb[dt->shared->u.compnd.nmembs].size = temp_type->shared->size;
                dt->shared->u.compnd.memb_size += temp_type->shared->size;

                
                dt->shared->u.compnd.memb[dt->shared->u.compnd.nmembs].type = temp_type;

                
                if (dt->shared->u.compnd.nmembs > 0 &&
                    dt->shared->u.compnd.memb[dt->shared->u.compnd.nmembs].offset < max_memb_pos) {
                    for (unsigned u = 0; u < dt->shared->u.compnd.nmembs; u++)
                        if ((dt->shared->u.compnd.memb[dt->shared->u.compnd.nmembs].offset >=
                                 dt->shared->u.compnd.memb[u].offset &&
                             dt->shared->u.compnd.memb[dt->shared->u.compnd.nmembs].offset <
                                 (dt->shared->u.compnd.memb[u].offset + dt->shared->u.compnd.memb[u].size)) ||
                            (dt->shared->u.compnd.memb[dt->shared->u.compnd.nmembs].offset <
                                 dt->shared->u.compnd.memb[u].offset &&
                             (dt->shared->u.compnd.memb[dt->shared->u.compnd.nmembs].offset +
                              dt->shared->u.compnd.memb[dt->shared->u.compnd.nmembs].size) >
                                 dt->shared->u.compnd.memb[u].offset))
                            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTDECODE, FAIL,
                                        "member overlaps with previous member");
                }

                
                max_memb_pos =
                    MAX(max_memb_pos, (dt->shared->u.compnd.memb[dt->shared->u.compnd.nmembs].offset +
                                       dt->shared->u.compnd.memb[dt->shared->u.compnd.nmembs].size));
            }

            
            H5T__update_packed(dt);

            
            if (version < upgrade_to) {
                version = upgrade_to;
                if (H5T__upgrade_version(dt, upgrade_to) < 0)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTSET, FAIL, "can't upgrade compound encoding version");
                
            }

            
            H5O_DTYPE_CHECK_VERSION(dt, version, max_version, ioflags, "compound", FAIL)
        } break;

        case H5T_REFERENCE:
            
            dt->shared->u.atomic.order   = H5T_ORDER_NONE;
            dt->shared->u.atomic.prec    = 8 * dt->shared->size;
            dt->shared->u.atomic.offset  = 0;
            dt->shared->u.atomic.lsb_pad = H5T_PAD_ZERO;
            dt->shared->u.atomic.msb_pad = H5T_PAD_ZERO;

            
            dt->shared->u.atomic.u.r.rtype = (H5R_type_t)(flags & 0x0f);
            if (dt->shared->u.atomic.u.r.rtype <= H5R_BADTYPE ||
                dt->shared->u.atomic.u.r.rtype >= H5R_MAXTYPE)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTDECODE, FAIL, "invalid reference type");

            
            if (dt->shared->u.atomic.u.r.rtype == H5R_OBJECT2 ||
                dt->shared->u.atomic.u.r.rtype == H5R_DATASET_REGION2 ||
                dt->shared->u.atomic.u.r.rtype == H5R_ATTR) {
                dt->shared->u.atomic.u.r.opaque  = true;
                dt->shared->u.atomic.u.r.version = (unsigned)((flags >> 4) & 0x0f);
                if (dt->shared->u.atomic.u.r.version != H5R_ENCODE_VERSION)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTDECODE, FAIL, "reference version does not match");
            }
            else
                dt->shared->u.atomic.u.r.opaque = false;

            
            dt->shared->force_conv = true;

            
            if (H5T_set_loc(dt, NULL, H5T_LOC_BADLOC) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "invalid datatype location");
            break;

        case H5T_ENUM: {
            unsigned nmembs;

            
            nmembs = flags & 0xffff;
            if (NULL == (dt->shared->parent = H5T__alloc()))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "can't allocate parent datatype");
            if (H5O__dtype_decode_helper(ioflags, pp, dt->shared->parent, skip, p_end) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTDECODE, FAIL, "unable to decode parent datatype");
            if (dt->shared->parent->shared->size != dt->shared->size)
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADSIZE, FAIL, "ENUM datatype size does not match parent");

            
            H5O_DTYPE_CHECK_VERSION(dt, version, dt->shared->parent->shared->version, ioflags, "enum", FAIL)

            
            if (NULL == (dt->shared->u.enumer.name = (char **)H5MM_calloc(nmembs * sizeof(char *))) ||
                NULL == (dt->shared->u.enumer.value =
                             (uint8_t *)H5MM_calloc(nmembs * dt->shared->parent->shared->size)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, FAIL, "memory allocation failed");
            dt->shared->u.enumer.nalloc = nmembs;

            
            for (dt->shared->u.enumer.nmembs = 0; dt->shared->u.enumer.nmembs < nmembs;
                 dt->shared->u.enumer.nmembs++) {

                size_t actual_name_length = 0; 

                
                if (!skip) {
                    

                    size_t max = (size_t)(p_end - *pp + 1); 

                    actual_name_length = strnlen((const char *)*pp, max);
                    if (actual_name_length == max)
                        HGOTO_ERROR(H5E_OHDR, H5E_NOSPACE, FAIL, "enum name not null terminated");
                }
                else {
                    
                    actual_name_length = strlen((const char *)*pp);
                }
                if (0 == actual_name_length)
                    HGOTO_ERROR(H5E_OHDR, H5E_BADSIZE, FAIL, "0 length enum name");

                if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, actual_name_length, p_end))
                    HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
                if (NULL == (dt->shared->u.enumer.name[dt->shared->u.enumer.nmembs] =
                                 H5MM_xstrdup((const char *)*pp)))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_CANTCOPY, FAIL, "can't duplicate enum name string");

                
                if (version >= H5O_DTYPE_VERSION_3) {
                    
                    if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, actual_name_length + 1, p_end))
                        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL,
                                    "ran off end of input buffer while decoding");
                    *pp += actual_name_length + 1;
                }
                else {
                    
                    if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, ((actual_name_length + 8) / 8) * 8, p_end))
                        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL,
                                    "ran off end of input buffer while decoding");
                    *pp += ((actual_name_length + 8) / 8) * 8;
                }
            }
            if (dt->shared->u.enumer.nmembs != nmembs)
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "incorrect number of enum members decoded");

            
            if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, nmembs * dt->shared->parent->shared->size, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
            H5MM_memcpy(dt->shared->u.enumer.value, *pp, nmembs * dt->shared->parent->shared->size);
            *pp += nmembs * dt->shared->parent->shared->size;
        } break;

        case H5T_VLEN:
            
            
            dt->shared->u.vlen.type = (H5T_vlen_type_t)(flags & 0x0f);
            if (dt->shared->u.vlen.type == H5T_VLEN_STRING) {
                dt->shared->u.vlen.pad  = (H5T_str_t)((flags >> 4) & 0x0f);
                dt->shared->u.vlen.cset = (H5T_cset_t)((flags >> 8) & 0x0f);
            }

            
            if (NULL == (dt->shared->parent = H5T__alloc()))
                HGOTO_ERROR(H5E_DATATYPE, H5E_NOSPACE, FAIL, "memory allocation failed");
            if (H5O__dtype_decode_helper(ioflags, pp, dt->shared->parent, skip, p_end) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTDECODE, FAIL, "unable to decode VL parent type");

            
            H5O_DTYPE_CHECK_VERSION(dt, version, dt->shared->parent->shared->version, ioflags, "vlen", FAIL)

            dt->shared->force_conv = true;

            
            if (H5T_set_loc(dt, NULL, H5T_LOC_BADLOC) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "invalid datatype location");
            break;

        case H5T_ARRAY:
            
            
            if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, 1, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
            dt->shared->u.array.ndims = *(*pp)++;

            
            if (dt->shared->u.array.ndims > H5S_MAX_RANK)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTLOAD, FAIL, "too many dimensions for array datatype");

            
            if (version < H5O_DTYPE_VERSION_3) {
                if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, 3, p_end))
                    HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
                *pp += 3;
            }

            
            dt->shared->u.array.nelem = 1;
            if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, (dt->shared->u.array.ndims * 4), p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
            for (unsigned u = 0; u < dt->shared->u.array.ndims; u++) {
                UINT32DECODE(*pp, dt->shared->u.array.dim[u]);
                dt->shared->u.array.nelem *= dt->shared->u.array.dim[u];
            }

            
            if (version < H5O_DTYPE_VERSION_3) {
                if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *pp, (dt->shared->u.array.ndims * 4), p_end))
                    HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, FAIL, "ran off end of input buffer while decoding");
                *pp += dt->shared->u.array.ndims * 4;
            }

            
            if (NULL == (dt->shared->parent = H5T__alloc()))
                HGOTO_ERROR(H5E_DATATYPE, H5E_NOSPACE, FAIL, "memory allocation failed");
            if (H5O__dtype_decode_helper(ioflags, pp, dt->shared->parent, skip, p_end) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTDECODE, FAIL, "unable to decode array parent type");

            
            H5O_DTYPE_CHECK_VERSION(dt, version, dt->shared->parent->shared->version, ioflags, "array", FAIL)

            
            H5O_DTYPE_CHECK_VERSION(dt, version, H5O_DTYPE_VERSION_2, ioflags, "array", FAIL)

            
            if (dt->shared->parent->shared->force_conv == true)
                dt->shared->force_conv = true;
            break;

        case H5T_COMPLEX: {
            bool homogeneous;

            

            
            homogeneous = (bool)(flags & 0x01);

            if (!homogeneous)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                            "heterogeneous complex number datatypes are currently unsupported");

            
            dt->shared->u.cplx.form = (H5T_complex_form_t)((flags >> 1) & 0x03);

            if (dt->shared->u.cplx.form != H5T_COMPLEX_RECTANGULAR)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                            "only complex number datatypes in rectangular form are currently supported");

            
            if ((flags >> 3) != 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADVALUE, FAIL,
                            "invalid flag bits set for complex number datatype");

            
            if (NULL == (dt->shared->parent = H5T__alloc()))
                HGOTO_ERROR(H5E_DATATYPE, H5E_NOSPACE, FAIL,
                            "unable to allocate complex number base datatype");
            if (H5O__dtype_decode_helper(ioflags, pp, dt->shared->parent, skip, p_end) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTDECODE, FAIL,
                            "unable to decode complex number base datatype");

            
            H5O_DTYPE_CHECK_VERSION(dt, version, dt->shared->parent->shared->version, ioflags, "complex",
                                    FAIL)

            
            H5O_DTYPE_CHECK_VERSION(dt, version, H5O_DTYPE_VERSION_5, ioflags, "complex", FAIL)

            break;
        }

        case H5T_NO_CLASS:
        case H5T_NCLASSES:
        default:
            HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "unknown datatype class found");
    }

    
    if (H5T_is_numeric_with_unusual_unused_bits(dt))
        
        if (!(*ioflags & H5O_DECODEIO_RFIC_UNUBNT))
            HGOTO_ERROR(
                H5E_DATATYPE, H5E_BADVALUE, FAIL,
                "datatype has unusually large # of unused bits (prec = %llu bits, size = %llu bytes), possibly "
                "corrupted file. See documentation for H5Pset_relax_file_integrity_checks for details.",
                (unsigned long long)dt->shared->u.atomic.prec, (unsigned long long)dt->shared->size);

done:
    
    if (ret_value < 0)
        
        if (H5T__free(dt) < 0)
            HDONE_ERROR(H5E_DATATYPE, H5E_CANTRELEASE, FAIL, "can't release datatype info");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__dtype_encode_helper(uint8_t **pp, const H5T_t *dt)
{
    unsigned flags = 0;
    uint8_t *hdr   = (uint8_t *)*pp;
    unsigned i;
    size_t   n, z;
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(pp && *pp);
    assert(dt);

    
    *pp += 4;
    UINT32ENCODE(*pp, dt->shared->size);

    switch (dt->shared->type) {
        case H5T_INTEGER:
            
            switch (dt->shared->u.atomic.order) {
                case H5T_ORDER_LE:
                    break; 

                case H5T_ORDER_BE:
                    flags |= 0x01;
                    break;

                case H5T_ORDER_ERROR:
                case H5T_ORDER_VAX:
                case H5T_ORDER_MIXED:
                case H5T_ORDER_NONE:
                default:
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                "byte order is not supported in file format yet");
            } 

            switch (dt->shared->u.atomic.lsb_pad) {
                case H5T_PAD_ZERO:
                    break; 

                case H5T_PAD_ONE:
                    flags |= 0x02;
                    break;

                case H5T_PAD_ERROR:
                case H5T_PAD_BACKGROUND:
                case H5T_NPAD:
                default:
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                "bit padding is not supported in file format yet");
            } 

            switch (dt->shared->u.atomic.msb_pad) {
                case H5T_PAD_ZERO:
                    break; 

                case H5T_PAD_ERROR:
                case H5T_PAD_BACKGROUND:
                case H5T_NPAD:
                case H5T_PAD_ONE:
                    flags |= 0x04;
                    break;

                default:
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                "bit padding is not supported in file format yet");
            } 

            switch (dt->shared->u.atomic.u.i.sign) {
                case H5T_SGN_NONE:
                    break; 

                case H5T_SGN_2:
                    flags |= 0x08;
                    break;

                case H5T_SGN_ERROR:
                case H5T_NSGN:
                default:
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                "sign scheme is not supported in file format yet");
            } 

            UINT16ENCODE(*pp, dt->shared->u.atomic.offset);
            UINT16ENCODE(*pp, dt->shared->u.atomic.prec);
            break;

        case H5T_FLOAT:
            
            switch (dt->shared->u.atomic.order) {
                case H5T_ORDER_LE:
                    break; 

                case H5T_ORDER_BE:
                    flags |= 0x01;
                    break;

                case H5T_ORDER_VAX: 
                    flags |= 0x41;
                    assert(dt->shared->version >= H5O_DTYPE_VERSION_3);
                    break;

                case H5T_ORDER_MIXED:
                case H5T_ORDER_ERROR:
                case H5T_ORDER_NONE:
                default:
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                "byte order is not supported in file format yet");
            } 

            switch (dt->shared->u.atomic.lsb_pad) {
                case H5T_PAD_ZERO:
                    break; 

                case H5T_PAD_ONE:
                    flags |= 0x02;
                    break;

                case H5T_PAD_ERROR:
                case H5T_PAD_BACKGROUND:
                case H5T_NPAD:
                default:
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                "bit padding is not supported in file format yet");
            } 

            switch (dt->shared->u.atomic.msb_pad) {
                case H5T_PAD_ZERO:
                    break; 

                case H5T_PAD_ONE:
                    flags |= 0x04;
                    break;

                case H5T_PAD_ERROR:
                case H5T_PAD_BACKGROUND:
                case H5T_NPAD:
                default:
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                "bit padding is not supported in file format yet");
            } 

            switch (dt->shared->u.atomic.u.f.pad) {
                case H5T_PAD_ZERO:
                    break; 

                case H5T_PAD_ONE:
                    flags |= 0x08;
                    break;

                case H5T_PAD_ERROR:
                case H5T_PAD_BACKGROUND:
                case H5T_NPAD:
                default:
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                "bit padding is not supported in file format yet");
            } 

            switch (dt->shared->u.atomic.u.f.norm) {
                case H5T_NORM_NONE:
                    break; 

                case H5T_NORM_MSBSET:
                    flags |= 0x10;
                    break;

                case H5T_NORM_IMPLIED:
                    flags |= 0x20;
                    break;

                case H5T_NORM_ERROR:
                default:
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                "normalization scheme is not supported in file format yet");
            } 

            flags = (unsigned)(flags | ((dt->shared->u.atomic.u.f.sign << 8) & 0xff00));
            UINT16ENCODE(*pp, dt->shared->u.atomic.offset);
            UINT16ENCODE(*pp, dt->shared->u.atomic.prec);
            assert(dt->shared->u.atomic.u.f.epos <= 255);
            *(*pp)++ = (uint8_t)(dt->shared->u.atomic.u.f.epos);
            assert(dt->shared->u.atomic.u.f.esize <= 255);
            *(*pp)++ = (uint8_t)(dt->shared->u.atomic.u.f.esize);
            assert(dt->shared->u.atomic.u.f.mpos <= 255);
            *(*pp)++ = (uint8_t)(dt->shared->u.atomic.u.f.mpos);
            assert(dt->shared->u.atomic.u.f.msize <= 255);
            *(*pp)++ = (uint8_t)(dt->shared->u.atomic.u.f.msize);
            UINT32ENCODE(*pp, dt->shared->u.atomic.u.f.ebias);
            break;

        case H5T_TIME: 
            switch (dt->shared->u.atomic.order) {
                case H5T_ORDER_LE:
                    break; 

                case H5T_ORDER_BE:
                    flags |= 0x01;
                    break;

                case H5T_ORDER_VAX:
                case H5T_ORDER_MIXED:
                case H5T_ORDER_ERROR:
                case H5T_ORDER_NONE:
                default:
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                "byte order is not supported in file format yet");
            } 
            UINT16ENCODE(*pp, dt->shared->u.atomic.prec);
            break;

        case H5T_STRING:
            
            assert(dt->shared->u.atomic.order == H5T_ORDER_NONE);
            assert(dt->shared->u.atomic.prec == 8 * dt->shared->size);
            assert(dt->shared->u.atomic.offset == 0);
            assert(dt->shared->u.atomic.lsb_pad == H5T_PAD_ZERO);
            assert(dt->shared->u.atomic.msb_pad == H5T_PAD_ZERO);

            flags = (unsigned)(flags | (dt->shared->u.atomic.u.s.pad & 0x0f));
            flags = (unsigned)(flags | ((((unsigned)dt->shared->u.atomic.u.s.cset) & 0x0f) << 4));
            break;

        case H5T_BITFIELD:
            
            switch (dt->shared->u.atomic.order) {
                case H5T_ORDER_LE:
                    break; 

                case H5T_ORDER_BE:
                    flags |= 0x01;
                    break;

                case H5T_ORDER_VAX:
                case H5T_ORDER_MIXED:
                case H5T_ORDER_ERROR:
                case H5T_ORDER_NONE:
                default:
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                "byte order is not supported in file format yet");
            } 

            switch (dt->shared->u.atomic.lsb_pad) {
                case H5T_PAD_ZERO:
                    break; 

                case H5T_PAD_ONE:
                    flags |= 0x02;
                    break;

                case H5T_PAD_ERROR:
                case H5T_PAD_BACKGROUND:
                case H5T_NPAD:
                default:
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                "bit padding is not supported in file format yet");
            } 

            switch (dt->shared->u.atomic.msb_pad) {
                case H5T_PAD_ZERO:
                    break; 

                case H5T_PAD_ONE:
                    flags |= 0x04;
                    break;

                case H5T_PAD_ERROR:
                case H5T_PAD_BACKGROUND:
                case H5T_NPAD:
                default:
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                "bit padding is not supported in file format yet");
            } 

            UINT16ENCODE(*pp, dt->shared->u.atomic.offset);
            UINT16ENCODE(*pp, dt->shared->u.atomic.prec);
            break;

        case H5T_OPAQUE:
            
            {
                size_t aligned;

                z       = strlen(dt->shared->u.opaque.tag);
                aligned = (z + 7) & (H5T_OPAQUE_TAG_MAX - 8);
                flags   = (unsigned)(flags | aligned);
                H5MM_memcpy(*pp, dt->shared->u.opaque.tag, MIN(z, aligned));
                for (n = MIN(z, aligned); n < aligned; n++)
                    (*pp)[n] = 0;
                *pp += aligned;
            }
            break;

        case H5T_COMPOUND: {
            unsigned offset_nbytes; 

            
            offset_nbytes = H5VM_limit_enc_size((uint64_t)dt->shared->size);

            
            flags = dt->shared->u.compnd.nmembs & 0xffff;
            for (i = 0; i < dt->shared->u.compnd.nmembs; i++) {
                
                
                assert(dt->shared->u.compnd.memb[i].type->shared->type != H5T_ARRAY ||
                       dt->shared->version >= H5O_DTYPE_VERSION_2);

                
                assert(dt->shared->version >= dt->shared->u.compnd.memb[i].type->shared->version);

                
                strcpy((char *)(*pp), dt->shared->u.compnd.memb[i].name);

                
                n = strlen(dt->shared->u.compnd.memb[i].name);
                if (dt->shared->version >= H5O_DTYPE_VERSION_3)
                    *pp += n + 1;
                else {
                    
                    for (z = n + 1; z % 8; z++)
                        (*pp)[z] = '\0';
                    *pp += z;
                } 

                
                
                if (dt->shared->version >= H5O_DTYPE_VERSION_3)
                    UINT32ENCODE_VAR(*pp, (uint32_t)dt->shared->u.compnd.memb[i].offset, offset_nbytes);
                else
                    UINT32ENCODE(*pp, dt->shared->u.compnd.memb[i].offset);

                
                if (dt->shared->version == H5O_DTYPE_VERSION_1) {
                    unsigned j;

                    
                    *(*pp)++ = 0;

                    
                    *(*pp)++ = 0;
                    *(*pp)++ = 0;
                    *(*pp)++ = 0;

                    
                    UINT32ENCODE(*pp, 0);

                    
                    UINT32ENCODE(*pp, 0);

                    
                    for (j = 0; j < 4; j++)
                        UINT32ENCODE(*pp, 0);
                } 

                
                if (H5O__dtype_encode_helper(pp, dt->shared->u.compnd.memb[i].type) < 0)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTENCODE, FAIL, "unable to encode member type");
            } 
        } break;

        case H5T_REFERENCE:
            flags |= (dt->shared->u.atomic.u.r.rtype & 0x0f);
            if (dt->shared->u.atomic.u.r.opaque)
                flags = (unsigned)(flags | (((unsigned)dt->shared->u.atomic.u.r.version & 0x0f) << 4));
            break;

        case H5T_ENUM:
            
            assert(dt->shared->version >= dt->shared->parent->shared->version);

            
            flags = dt->shared->u.enumer.nmembs & 0xffff;

            
            if (H5O__dtype_encode_helper(pp, dt->shared->parent) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTENCODE, FAIL, "unable to encode parent datatype");

            
            for (i = 0; i < dt->shared->u.enumer.nmembs; i++) {
                
                strcpy((char *)(*pp), dt->shared->u.enumer.name[i]);

                
                n = strlen(dt->shared->u.enumer.name[i]);
                if (dt->shared->version >= H5O_DTYPE_VERSION_3)
                    *pp += n + 1;
                else {
                    
                    for (z = n + 1; z % 8; z++)
                        (*pp)[z] = '\0';
                    *pp += z;
                } 
            }     

            
            H5MM_memcpy(*pp, dt->shared->u.enumer.value,
                        dt->shared->u.enumer.nmembs * dt->shared->parent->shared->size);
            *pp += dt->shared->u.enumer.nmembs * dt->shared->parent->shared->size;
            break;

        case H5T_VLEN: 
            
            assert(dt->shared->version >= dt->shared->parent->shared->version);

            flags |= (dt->shared->u.vlen.type & 0x0f);
            if (dt->shared->u.vlen.type == H5T_VLEN_STRING) {
                flags = (unsigned)(flags | (((unsigned)dt->shared->u.vlen.pad & 0x0f) << 4));
                flags = (unsigned)(flags | (((unsigned)dt->shared->u.vlen.cset & 0x0f) << 8));
            } 

            
            if (H5O__dtype_encode_helper(pp, dt->shared->parent) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTENCODE, FAIL, "unable to encode VL parent type");
            break;

        case H5T_ARRAY: 
            
            assert(dt->shared->u.array.ndims <= H5S_MAX_RANK);

            
            assert(dt->shared->version >= H5O_DTYPE_VERSION_2);

            
            assert(dt->shared->version >= dt->shared->parent->shared->version);

            
            assert(dt->shared->u.array.ndims <= UCHAR_MAX);
            *(*pp)++ = (uint8_t)dt->shared->u.array.ndims;

            
            if (dt->shared->version < H5O_DTYPE_VERSION_3) {
                
                *(*pp)++ = '\0';
                *(*pp)++ = '\0';
                *(*pp)++ = '\0';
            } 

            
            for (i = 0; i < (unsigned)dt->shared->u.array.ndims; i++)
                UINT32ENCODE(*pp, dt->shared->u.array.dim[i]);

            
            if (dt->shared->version < H5O_DTYPE_VERSION_3) {
                
                for (i = 0; i < (unsigned)dt->shared->u.array.ndims; i++)
                    UINT32ENCODE(*pp, i);
            } 

            
            if (H5O__dtype_encode_helper(pp, dt->shared->parent) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTENCODE, FAIL, "unable to encode VL parent type");
            break;

        case H5T_COMPLEX:
            
            assert(dt->shared->version >= H5O_DTYPE_VERSION_5);

            
            assert(dt->shared->version >= dt->shared->parent->shared->version);

            if (dt->shared->u.cplx.form != H5T_COMPLEX_RECTANGULAR)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTENCODE, FAIL,
                            "complex number datatypes not in rectangular form are currently unsupported");

            
            flags |= 0x01;

            
            flags = (unsigned)(flags | (((unsigned)dt->shared->u.cplx.form & 0x03) << 1));

            
            if (H5O__dtype_encode_helper(pp, dt->shared->parent) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTENCODE, FAIL,
                            "unable to encode complex number base datatype");

            break;

        case H5T_NO_CLASS:
        case H5T_NCLASSES:
        default:
            
            break;
    } 

    
    *hdr++ = (uint8_t)(((unsigned)(dt->shared->type) & 0x0f) | (dt->shared->version << 4));
    *hdr++ = (uint8_t)((flags >> 0) & 0xff);
    *hdr++ = (uint8_t)((flags >> 8) & 0xff);
    *hdr++ = (uint8_t)((flags >> 16) & 0xff);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5O__dtype_decode(H5F_t *f, H5O_t *open_oh, unsigned H5_ATTR_UNUSED mesg_flags, unsigned *ioflags ,
                  size_t p_size, const uint8_t *p)
{
    bool           skip;
    H5T_t         *dt        = NULL;
    const uint8_t *p_end     = p + p_size - 1;
    void          *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(p);

    
    if (NULL == (dt = H5T__alloc()))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    skip = (p_size == SIZE_MAX ? true : false);

    
    if (open_oh) {
        if (H5O_SIZEOF_CHKSUM_OH(open_oh) > 0 ||
            (f && (H5F_RFIC_FLAGS(f) & H5F_RFIC_UNUSUAL_NUM_UNUSED_NUMERIC_BITS)))
            *ioflags |= H5O_DECODEIO_RFIC_UNUBNT;
    }
    else
        
        *ioflags |= H5O_DECODEIO_RFIC_UNUBNT;

    
    if (H5O__dtype_decode_helper(ioflags, &p, dt, skip, p_end) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTDECODE, NULL, "can't decode type");

    
    ret_value = dt;

done:
    
    if (!ret_value)
        
        if (H5T_close_real(dt) < 0)
            HDONE_ERROR(H5E_DATATYPE, H5E_CANTRELEASE, NULL, "can't release datatype info");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__dtype_encode(H5F_t H5_ATTR_UNUSED *f, uint8_t *p, const void *mesg)
{
    const H5T_t *dt        = (const H5T_t *)mesg;
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(p);
    assert(dt);

    
    if (H5O__dtype_encode_helper(&p, dt) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTENCODE, FAIL, "can't encode type");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5O__dtype_copy(const void *_src, void *_dst)
{
    const H5T_t *src = (const H5T_t *)_src;
    H5T_t       *dst;
    void        *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(src);

    
    if (NULL == (dst = H5T_copy(src, H5T_COPY_ALL)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "can't copy type");

    
    if (_dst) {
        *((H5T_t *)_dst) = *dst;
        dst              = H5FL_FREE(H5T_t, dst);
        dst              = (H5T_t *)_dst;
    } 

    
    ret_value = dst;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5O__dtype_size(const H5F_t *f, const void *_mesg)
{
    const H5T_t *dt = (const H5T_t *)_mesg;
    unsigned     u;             
    size_t       ret_value = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(f);
    assert(dt);

    
    ret_value = 4 + 
                4;  

    
    switch (dt->shared->type) {
        case H5T_INTEGER:
            ret_value += 4;
            break;

        case H5T_FLOAT:
            ret_value += 12;
            break;

        case H5T_TIME:
            ret_value += 2;
            break;

        case H5T_BITFIELD:
            ret_value += 4;
            break;

        case H5T_OPAQUE:
            ret_value += (strlen(dt->shared->u.opaque.tag) + 7) & (H5T_OPAQUE_TAG_MAX - 8);
            break;

        case H5T_COMPOUND: {
            unsigned offset_nbytes; 

            
            offset_nbytes = H5VM_limit_enc_size((uint64_t)dt->shared->size);

            
            for (u = 0; u < dt->shared->u.compnd.nmembs; u++) {
                size_t name_len; 

                
                name_len = strlen(dt->shared->u.compnd.memb[u].name);

                
                if (dt->shared->version >= H5O_DTYPE_VERSION_3)
                    ret_value += name_len + 1;
                else
                    ret_value += ((name_len + 8) / 8) * 8;

                
                
                if (dt->shared->version >= H5O_DTYPE_VERSION_3)
                    ret_value += offset_nbytes; 
                else if (dt->shared->version == H5O_DTYPE_VERSION_2)
                    ret_value += 4; 
                else
                    ret_value += 4 + 
                                 1 + 
                                 3 + 
                                 4 + 
                                 4 + 
                                 16; 
                ret_value += H5O__dtype_size(f, dt->shared->u.compnd.memb[u].type);
            } 
        } break;

        case H5T_ENUM:
            ret_value += H5O__dtype_size(f, dt->shared->parent);
            for (u = 0; u < dt->shared->u.enumer.nmembs; u++) {
                size_t name_len; 

                
                name_len = strlen(dt->shared->u.enumer.name[u]);

                
                if (dt->shared->version >= H5O_DTYPE_VERSION_3)
                    ret_value += name_len + 1;
                else
                    ret_value += ((name_len + 8) / 8) * 8;
            } 
            ret_value += dt->shared->u.enumer.nmembs * dt->shared->parent->shared->size;
            break;

        case H5T_VLEN:
            ret_value += H5O__dtype_size(f, dt->shared->parent);
            break;

        case H5T_ARRAY:
            ret_value += 1; 
            if (dt->shared->version < H5O_DTYPE_VERSION_3)
                ret_value += 3;                         
            ret_value += 4 * dt->shared->u.array.ndims; 
            if (dt->shared->version < H5O_DTYPE_VERSION_3)
                ret_value += 4 * dt->shared->u.array.ndims; 
            ret_value += H5O__dtype_size(f, dt->shared->parent);
            break;

        case H5T_COMPLEX:
            ret_value += H5O__dtype_size(f, dt->shared->parent);
            break;

        case H5T_NO_CLASS:
        case H5T_STRING:
        case H5T_REFERENCE:
        case H5T_NCLASSES:
        default:
            
            break;
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__dtype_reset(void *_mesg)
{
    H5T_t *dt = (H5T_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    if (dt)
        H5T__free(dt);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__dtype_free(void *mesg)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(mesg);

    
    if (H5T_close_real((H5T_t *)mesg) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTFREE, FAIL, "unable to free datatype");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__dtype_set_share(void *_mesg , const H5O_shared_t *sh)
{
    H5T_t *dt        = (H5T_t *)_mesg;
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(dt);
    assert(sh);

    
    assert(sh->type <= H5O_SHARE_TYPE_HERE);

    
    assert(sh->type == H5O_SHARE_TYPE_COMMITTED ||
           (dt->shared->state != H5T_STATE_OPEN && dt->shared->state != H5T_STATE_NAMED));

    
    if (H5O_set_shared(&(dt->sh_loc), sh) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, FAIL, "unable to copy shared message info");

    
    if (sh->type == H5O_SHARE_TYPE_COMMITTED) {
        dt->shared->state = H5T_STATE_NAMED;

        
        if (H5O_loc_reset(&(dt->oloc)) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to reset location");
        dt->oloc.file = sh->file;
        dt->oloc.addr = sh->u.loc.oh_addr;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static htri_t
H5O__dtype_can_share(const void *_mesg)
{
    const H5T_t *mesg = (const H5T_t *)_mesg;
    htri_t       tri_ret;
    htri_t       ret_value = true;

    FUNC_ENTER_PACKAGE

    assert(mesg);

    
    if ((tri_ret = H5T_is_immutable(mesg)) > 0)
        HGOTO_DONE(false);
    else if (tri_ret < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_BADTYPE, FAIL, "can't tell if datatype is immutable");

    
    if ((tri_ret = H5T_is_named(mesg)) > 0)
        HGOTO_DONE(false);
    else if (tri_ret < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_BADTYPE, FAIL, "can't tell if datatype is shared");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__dtype_pre_copy_file(H5F_t *file_src, const void *mesg_src, bool H5_ATTR_UNUSED *deleted,
                         const H5O_copy_t *cpy_info, void *_udata)
{
    const H5T_t        *dt_src    = (const H5T_t *)mesg_src;      
    H5D_copy_file_ud_t *udata     = (H5D_copy_file_ud_t *)_udata; 
    herr_t              ret_value = SUCCEED;                      

    FUNC_ENTER_PACKAGE

    
    assert(file_src);
    assert(dt_src);
    assert(cpy_info);
    assert(cpy_info->file_dst);

    
    if (dt_src->shared->version > H5O_dtype_ver_bounds[H5F_HIGH_BOUND(cpy_info->file_dst)])
        HGOTO_ERROR(H5E_OHDR, H5E_BADRANGE, FAIL, "datatype message version out of bounds");

    
    if (udata) {
        
        if (NULL == (udata->src_dtype = H5T_copy(dt_src, H5T_COPY_TRANSIENT)))
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to copy");

        
        if (H5T_set_loc(udata->src_dtype, H5F_VOL_OBJ(file_src), H5T_LOC_DISK) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "cannot mark datatype on disk");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5O__dtype_copy_file(H5F_t H5_ATTR_UNUSED *file_src, const H5O_msg_class_t *mesg_type, void *native_src,
                     H5F_t *file_dst, bool H5_ATTR_UNUSED *recompute_size,
                     H5O_copy_t H5_ATTR_UNUSED *cpy_info, void H5_ATTR_UNUSED *udata)
{
    H5T_t *dst_mesg;         
    void  *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (dst_mesg = (H5T_t *)H5O__dtype_copy(native_src, NULL)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "unable to copy");

    
    if (H5T_set_loc(dst_mesg, H5F_VOL_OBJ(file_dst), H5T_LOC_DISK) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "unable to set location");

    ret_value = dst_mesg;

done:
    if (NULL == ret_value)
        H5O_msg_free(mesg_type->id, dst_mesg);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__dtype_shared_post_copy_upd(const H5O_loc_t H5_ATTR_UNUSED *src_oloc, const void H5_ATTR_UNUSED *mesg_src,
                                H5O_loc_t H5_ATTR_UNUSED *dst_oloc, void *mesg_dst,
                                H5O_copy_t H5_ATTR_UNUSED *cpy_info)
{
    H5T_t *dt_dst    = (H5T_t *)mesg_dst; 
    herr_t ret_value = SUCCEED;           

    FUNC_ENTER_PACKAGE

    if (dt_dst->sh_loc.type == H5O_SHARE_TYPE_COMMITTED) {
        assert(H5T_is_named(dt_dst));
        if (H5O_loc_reset(&(dt_dst->oloc)) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to reset location");
        dt_dst->oloc.file = dt_dst->sh_loc.file;
        dt_dst->oloc.addr = dt_dst->sh_loc.u.loc.oh_addr;
    } 
    else
        assert(!H5T_is_named(dt_dst));

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__dtype_debug(H5F_t *f, const void *mesg, FILE *stream, int indent, int fwidth)
{
    const H5T_t *dt = (const H5T_t *)mesg;
    const char  *s;
    char         buf[256];
    unsigned     i;
    size_t       k;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(dt);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    switch (dt->shared->type) {
        case H5T_INTEGER:
            s = "integer";
            break;

        case H5T_FLOAT:
            s = "floating-point";
            break;

        case H5T_TIME:
            s = "date and time";
            break;

        case H5T_STRING:
            s = "text string";
            break;

        case H5T_BITFIELD:
            s = "bit field";
            break;

        case H5T_OPAQUE:
            s = "opaque";
            break;

        case H5T_COMPOUND:
            s = "compound";
            break;

        case H5T_REFERENCE:
            s = "reference";
            break;

        case H5T_ENUM:
            s = "enum";
            break;

        case H5T_ARRAY:
            s = "array";
            break;

        case H5T_VLEN:
            s = "vlen";
            break;

        case H5T_COMPLEX:
            s = "complex number";
            break;

        case H5T_NO_CLASS:
        case H5T_NCLASSES:
        default:
            snprintf(buf, sizeof(buf), "H5T_CLASS_%d", (int)(dt->shared->type));
            s = buf;
            break;
    } 
    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Type class:", s);

    Rfprintf(stream, "%*s%-*s %lu byte%s\n", indent, "", fwidth, "Size:", (unsigned long)(dt->shared->size),
            1 == dt->shared->size ? "" : "s");

    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth, "Version:", dt->shared->version);

    if (H5T_COMPOUND == dt->shared->type) {
        Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
                "Number of members:", dt->shared->u.compnd.nmembs);
        for (i = 0; i < dt->shared->u.compnd.nmembs; i++) {
            snprintf(buf, sizeof(buf), "Member %u:", i);
            Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, buf, dt->shared->u.compnd.memb[i].name);
            Rfprintf(stream, "%*s%-*s %lu\n", indent + 3, "", MAX(0, fwidth - 3),
                    "Byte offset:", (unsigned long)(dt->shared->u.compnd.memb[i].offset));
            H5O__dtype_debug(f, dt->shared->u.compnd.memb[i].type, stream, indent + 3, MAX(0, fwidth - 3));
        } 
    }     
    else if (H5T_ENUM == dt->shared->type) {
        Rfprintf(stream, "%*s%s\n", indent, "", "Base type:");
        H5O__dtype_debug(f, dt->shared->parent, stream, indent + 3, MAX(0, fwidth - 3));
        Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
                "Number of members:", dt->shared->u.enumer.nmembs);
        for (i = 0; i < dt->shared->u.enumer.nmembs; i++) {
            snprintf(buf, sizeof(buf), "Member %u:", i);
            Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, buf, dt->shared->u.enumer.name[i]);
            Rfprintf(stream, "%*s%-*s 0x", indent, "", fwidth, "Raw bytes of value:");
            for (k = 0; k < dt->shared->parent->shared->size; k++)
                Rfprintf(stream, "%02x",
                        (unsigned)*((uint8_t *)dt->shared->u.enumer.value +
                                    (i * dt->shared->parent->shared->size) + k));
            Rfprintf(stream, "\n");
        } 
    }     
    else if (H5T_OPAQUE == dt->shared->type) {
        Rfprintf(stream, "%*s%-*s \"%s\"\n", indent, "", fwidth, "Tag:", dt->shared->u.opaque.tag);
    } 
    else if (H5T_REFERENCE == dt->shared->type) {
        Rfprintf(stream, "%*s%-*s\n", indent, "", fwidth, "Fix dumping reference types!");
    } 
    else if (H5T_STRING == dt->shared->type) {
        switch (dt->shared->u.atomic.u.s.cset) {
            case H5T_CSET_ASCII:
                s = "ASCII";
                break;

            case H5T_CSET_UTF8:
                s = "UTF-8";
                break;

            case H5T_CSET_RESERVED_2:
            case H5T_CSET_RESERVED_3:
            case H5T_CSET_RESERVED_4:
            case H5T_CSET_RESERVED_5:
            case H5T_CSET_RESERVED_6:
            case H5T_CSET_RESERVED_7:
            case H5T_CSET_RESERVED_8:
            case H5T_CSET_RESERVED_9:
            case H5T_CSET_RESERVED_10:
            case H5T_CSET_RESERVED_11:
            case H5T_CSET_RESERVED_12:
            case H5T_CSET_RESERVED_13:
            case H5T_CSET_RESERVED_14:
            case H5T_CSET_RESERVED_15:
                snprintf(buf, sizeof(buf), "H5T_CSET_RESERVED_%d", (int)(dt->shared->u.atomic.u.s.cset));
                s = buf;
                break;

            case H5T_CSET_ERROR:
            default:
                snprintf(buf, sizeof(buf), "Unknown character set: %d", (int)(dt->shared->u.atomic.u.s.cset));
                s = buf;
                break;
        } 
        Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Character Set:", s);

        switch (dt->shared->u.atomic.u.s.pad) {
            case H5T_STR_NULLTERM:
                s = "NULL Terminated";
                break;

            case H5T_STR_NULLPAD:
                s = "NULL Padded";
                break;

            case H5T_STR_SPACEPAD:
                s = "Space Padded";
                break;

            case H5T_STR_RESERVED_3:
            case H5T_STR_RESERVED_4:
            case H5T_STR_RESERVED_5:
            case H5T_STR_RESERVED_6:
            case H5T_STR_RESERVED_7:
            case H5T_STR_RESERVED_8:
            case H5T_STR_RESERVED_9:
            case H5T_STR_RESERVED_10:
            case H5T_STR_RESERVED_11:
            case H5T_STR_RESERVED_12:
            case H5T_STR_RESERVED_13:
            case H5T_STR_RESERVED_14:
            case H5T_STR_RESERVED_15:
                snprintf(buf, sizeof(buf), "H5T_STR_RESERVED_%d", (int)(dt->shared->u.atomic.u.s.pad));
                s = buf;
                break;

            case H5T_STR_ERROR:
            default:
                snprintf(buf, sizeof(buf), "Unknown string padding: %d", (int)(dt->shared->u.atomic.u.s.pad));
                s = buf;
                break;
        } 
        Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "String Padding:", s);
    } 
    else if (H5T_VLEN == dt->shared->type) {
        switch (dt->shared->u.vlen.type) {
            case H5T_VLEN_SEQUENCE:
                s = "sequence";
                break;

            case H5T_VLEN_STRING:
                s = "string";
                break;

            case H5T_VLEN_BADTYPE:
            case H5T_VLEN_MAXTYPE:
            default:
                snprintf(buf, sizeof(buf), "H5T_VLEN_%d", dt->shared->u.vlen.type);
                s = buf;
                break;
        } 
        Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Vlen type:", s);

        switch (dt->shared->u.vlen.loc) {
            case H5T_LOC_MEMORY:
                s = "memory";
                break;

            case H5T_LOC_DISK:
                s = "disk";
                break;

            case H5T_LOC_BADLOC:
            case H5T_LOC_MAXLOC:
            default:
                snprintf(buf, sizeof(buf), "H5T_LOC_%d", (int)dt->shared->u.vlen.loc);
                s = buf;
                break;
        } 
        Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Location:", s);

        
        if (dt->shared->u.vlen.type == H5T_VLEN_STRING) {
            switch (dt->shared->u.vlen.cset) {
                case H5T_CSET_ASCII:
                    s = "ASCII";
                    break;

                case H5T_CSET_UTF8:
                    s = "UTF-8";
                    break;

                case H5T_CSET_RESERVED_2:
                case H5T_CSET_RESERVED_3:
                case H5T_CSET_RESERVED_4:
                case H5T_CSET_RESERVED_5:
                case H5T_CSET_RESERVED_6:
                case H5T_CSET_RESERVED_7:
                case H5T_CSET_RESERVED_8:
                case H5T_CSET_RESERVED_9:
                case H5T_CSET_RESERVED_10:
                case H5T_CSET_RESERVED_11:
                case H5T_CSET_RESERVED_12:
                case H5T_CSET_RESERVED_13:
                case H5T_CSET_RESERVED_14:
                case H5T_CSET_RESERVED_15:
                    snprintf(buf, sizeof(buf), "H5T_CSET_RESERVED_%d", (int)(dt->shared->u.vlen.cset));
                    s = buf;
                    break;

                case H5T_CSET_ERROR:
                default:
                    snprintf(buf, sizeof(buf), "Unknown character set: %d", (int)(dt->shared->u.vlen.cset));
                    s = buf;
                    break;
            } 
            Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Character Set:", s);

            switch (dt->shared->u.vlen.pad) {
                case H5T_STR_NULLTERM:
                    s = "NULL Terminated";
                    break;

                case H5T_STR_NULLPAD:
                    s = "NULL Padded";
                    break;

                case H5T_STR_SPACEPAD:
                    s = "Space Padded";
                    break;

                case H5T_STR_RESERVED_3:
                case H5T_STR_RESERVED_4:
                case H5T_STR_RESERVED_5:
                case H5T_STR_RESERVED_6:
                case H5T_STR_RESERVED_7:
                case H5T_STR_RESERVED_8:
                case H5T_STR_RESERVED_9:
                case H5T_STR_RESERVED_10:
                case H5T_STR_RESERVED_11:
                case H5T_STR_RESERVED_12:
                case H5T_STR_RESERVED_13:
                case H5T_STR_RESERVED_14:
                case H5T_STR_RESERVED_15:
                    snprintf(buf, sizeof(buf), "H5T_STR_RESERVED_%d", (int)(dt->shared->u.vlen.pad));
                    s = buf;
                    break;

                case H5T_STR_ERROR:
                default:
                    snprintf(buf, sizeof(buf), "Unknown string padding: %d", (int)(dt->shared->u.vlen.pad));
                    s = buf;
                    break;
            } 
            Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "String Padding:", s);
        } 
    }     
    else if (H5T_ARRAY == dt->shared->type) {
        Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth, "Rank:", dt->shared->u.array.ndims);
        Rfprintf(stream, "%*s%-*s {", indent, "", fwidth, "Dim Size:");
        for (i = 0; i < dt->shared->u.array.ndims; i++)
            Rfprintf(stream, "%s%u", (i ? ", " : ""), (unsigned)dt->shared->u.array.dim[i]);
        Rfprintf(stream, "}\n");
        Rfprintf(stream, "%*s%s\n", indent, "", "Base type:");
        H5O__dtype_debug(f, dt->shared->parent, stream, indent + 3, MAX(0, fwidth - 3));
    } 
    else if (H5T_COMPLEX == dt->shared->type) {
        switch (dt->shared->u.cplx.form) {
            case H5T_COMPLEX_RECTANGULAR:
                Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Form:", "rectangular");
                break;
            case H5T_COMPLEX_POLAR:
                Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Form:", "polar");
                break;
            case H5T_COMPLEX_EXPONENTIAL:
                Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Form:", "exponential");
                break;
            default:
                Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Form:", "invalid");
                break;
        }

        Rfprintf(stream, "%*s%s\n", indent, "", "Base type:");
        H5O__dtype_debug(f, dt->shared->parent, stream, indent + 3, MAX(0, fwidth - 3));
    }
    else {
        switch (dt->shared->u.atomic.order) {
            case H5T_ORDER_LE:
                s = "little endian";
                break;

            case H5T_ORDER_BE:
                s = "big endian";
                break;

            case H5T_ORDER_VAX:
                s = "VAX";
                break;

            case H5T_ORDER_NONE:
                s = "none";
                break;

            case H5T_ORDER_MIXED:
                s = "mixed";
                break;

            case H5T_ORDER_ERROR:
            default:
                snprintf(buf, sizeof(buf), "H5T_ORDER_%d", dt->shared->u.atomic.order);
                s = buf;
                break;
        } 
        Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Byte order:", s);

        Rfprintf(stream, "%*s%-*s %lu bit%s\n", indent, "", fwidth,
                "Precision:", (unsigned long)(dt->shared->u.atomic.prec),
                1 == dt->shared->u.atomic.prec ? "" : "s");

        Rfprintf(stream, "%*s%-*s %lu bit%s\n", indent, "", fwidth,
                "Offset:", (unsigned long)(dt->shared->u.atomic.offset),
                1 == dt->shared->u.atomic.offset ? "" : "s");

        switch (dt->shared->u.atomic.lsb_pad) {
            case H5T_PAD_ZERO:
                s = "zero";
                break;

            case H5T_PAD_ONE:
                s = "one";
                break;

            case H5T_PAD_BACKGROUND:
                s = "background";
                break;

            case H5T_PAD_ERROR:
            case H5T_NPAD:
            default:
                s = "pad?";
                break;
        } 
        Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Low pad type:", s);

        switch (dt->shared->u.atomic.msb_pad) {
            case H5T_PAD_ZERO:
                s = "zero";
                break;

            case H5T_PAD_ONE:
                s = "one";
                break;

            case H5T_PAD_BACKGROUND:
                s = "background";
                break;

            case H5T_PAD_ERROR:
            case H5T_NPAD:
            default:
                s = "pad?";
                break;
        } 
        Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "High pad type:", s);

        if (H5T_FLOAT == dt->shared->type) {
            switch (dt->shared->u.atomic.u.f.pad) {
                case H5T_PAD_ZERO:
                    s = "zero";
                    break;

                case H5T_PAD_ONE:
                    s = "one";
                    break;

                case H5T_PAD_BACKGROUND:
                    s = "background";
                    break;

                case H5T_PAD_ERROR:
                case H5T_NPAD:
                default:
                    if (dt->shared->u.atomic.u.f.pad < 0)
                        snprintf(buf, sizeof(buf), "H5T_PAD_%d", -(dt->shared->u.atomic.u.f.pad));
                    else
                        snprintf(buf, sizeof(buf), "bit-%d", dt->shared->u.atomic.u.f.pad);
                    s = buf;
                    break;
            } 
            Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Internal pad type:", s);

            switch (dt->shared->u.atomic.u.f.norm) {
                case H5T_NORM_IMPLIED:
                    s = "implied";
                    break;

                case H5T_NORM_MSBSET:
                    s = "msb set";
                    break;

                case H5T_NORM_NONE:
                    s = "none";
                    break;

                case H5T_NORM_ERROR:
                default:
                    snprintf(buf, sizeof(buf), "H5T_NORM_%d", (int)(dt->shared->u.atomic.u.f.norm));
                    s = buf;
            } 
            Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Normalization:", s);

            Rfprintf(stream, "%*s%-*s %lu\n", indent, "", fwidth,
                    "Sign bit location:", (unsigned long)(dt->shared->u.atomic.u.f.sign));

            Rfprintf(stream, "%*s%-*s %lu\n", indent, "", fwidth,
                    "Exponent location:", (unsigned long)(dt->shared->u.atomic.u.f.epos));

            Rfprintf(stream, "%*s%-*s 0x%08lx\n", indent, "", fwidth,
                    "Exponent bias:", (unsigned long)(dt->shared->u.atomic.u.f.ebias));

            Rfprintf(stream, "%*s%-*s %lu\n", indent, "", fwidth,
                    "Exponent size:", (unsigned long)(dt->shared->u.atomic.u.f.esize));

            Rfprintf(stream, "%*s%-*s %lu\n", indent, "", fwidth,
                    "Mantissa location:", (unsigned long)(dt->shared->u.atomic.u.f.mpos));

            Rfprintf(stream, "%*s%-*s %lu\n", indent, "", fwidth,
                    "Mantissa size:", (unsigned long)(dt->shared->u.atomic.u.f.msize));

        } 
        else if (H5T_INTEGER == dt->shared->type) {
            switch (dt->shared->u.atomic.u.i.sign) {
                case H5T_SGN_NONE:
                    s = "none";
                    break;

                case H5T_SGN_2:
                    s = "2's comp";
                    break;

                case H5T_SGN_ERROR:
                case H5T_NSGN:
                default:
                    snprintf(buf, sizeof(buf), "H5T_SGN_%d", (int)(dt->shared->u.atomic.u.i.sign));
                    s = buf;
                    break;
            } 
            Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Sign scheme:", s);
        } 
    }     

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
