% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/buffer.r
\name{buffer,GRaster-method}
\alias{buffer,GRaster-method}
\alias{buffer}
\alias{buffer,GVector-method}
\alias{st_buffer,GVector-method}
\alias{st_buffer}
\title{Increase/decrease the size of a vector or around non-NA cells of a raster}
\usage{
\S4method{buffer}{GRaster}(
  x,
  width,
  unit = "meters",
  method = "Euclidean",
  background = 0,
  lowMemory = FALSE
)

\S4method{buffer}{GVector}(x, width, capstyle = "round", dissolve = TRUE)

\S4method{st_buffer}{GVector}(x, dist, endCapStyle = "round", dissolve = FALSE)
}
\arguments{
\item{x}{A \code{GRaster} or \code{GVector}.}

\item{width}{Numeric: For rasters -- Maximum distance cells must be from focal cells to be within the buffer. For rasters, if the buffering unit is \verb{"cells}", then to get \code{n} cell widths, use \code{n + epsilon}, where \code{epsilon} is a small number (e.g., 0.001). The larger the buffer, this smaller this must be to ensure just \code{n} cells are included.

For vectors, distance from the object to place the buffer. Negative values create "inside" buffers. Units are in the same units as the current coordinate reference system (e.g., degrees for WGS84 or NAD83, often meters for projected systems).}

\item{unit}{Character: Rasters -- Indicates the units of \code{width}. Can be one of:
\itemize{
\item \code{"cells"}: Units are numbers of cells.
\item \code{"meters"} (default), \code{"metres"}, or \code{"m"}
\item \code{"kilometers"} or \code{"km"}
\item \code{"feet"} or \code{"ft"}
\item \code{"miles"} or \code{"mi"}
\item \code{"nautical miles"} or \code{"nmi"}
}

Partial matching is used and case is ignored.}

\item{method}{Character: Rasters -- Only used if \code{units} is \code{"cells"}. Indicates the manner in which distances are calculated for adding of cells:
\itemize{
\item \code{"Euclidean"}: Euclidean distance (default)
\item \code{"Manhattan"}: "taxi-cab" distance
\item \code{"maximum"}: Maximum of the north-south and east-west distances between points.
}

Partial matching is used and case is ignored.}

\item{background}{Numeric: Rasters -- Value to assign to cells that are not \code{NA} and not part of the buffer (default is 0).}

\item{lowMemory}{Logical: Rasters -- Only used if buffering a raster and \code{units} is not \code{"meters"}. If \code{FALSE} (default) use faster, memory-intensive procedure. If \code{TRUE} then use the slower, low-memory version. To help decide which to use, consider using the low-memory version on a system with 1 GB of RAM for a raster larger than about 32000 x 32000 cells, or for a system with  with 8 GB of RAM a raster larger than about 90000 x 90000 cells.}

\item{capstyle, endCapStyle}{Character: Vectors -- Style for ending the "cap" of buffers around lines. Valid options include \code{"rounded"}, \code{"square"}, and "\code{flat}".}

\item{dissolve}{Logical (\code{GVector}s): If \code{TRUE} (default), dissolve all buffers after creation. If \code{FALSE}, construct a buffer for each geometry. Note that overlapping buffers can cause this function to fail because it creates a topologically ambiguous polygon. Thus, using \code{dissolve = TRUE} is recommended.}

\item{dist}{Vectors -- Same as \code{width}.}
}
\value{
A \code{GRaster} or a \code{GVector}.
}
\description{
Buffers can be constructed for \code{GRaster}s or \code{GVector}s. For rasters, the \code{buffer()} function creates a buffer around non-\code{NA} cells. The output will be a raster. For vectors, the \code{buffer()} and \code{st_buffer()} functions create a vector polygon larger or smaller than the focal vector.
}
\details{
Note that in some cases, topologically incorrect vectors can be created when buffering. This can arise when buffers intersect to create intersections that technically belong to two or more geometries. This issue can be resolved by dissolving borders between buffered geometries using \code{dissolve = TRUE}, but as of now, there is no fix if you do not want to dissolve geometries. A workaround would be to create a different \code{GVector} for each geometry, and then buffer each individually :(.
}
\examples{
if (grassStarted()) {

# Setup
library(sf)
library(terra)

# Elevation raster, rivers vector
madElev <- fastData("madElev")
madRivers <- fastData("madRivers")

# Convert a SpatRaster to a GRaster, and sf to a GVector
elev <- fast(madElev)
rivers <- fast(madRivers)

### Buffer a raster by a given distance:
buffByDist <- buffer(elev, width = 2000) # 2000-m buffer
plot(buffByDist, legend = FALSE)
plot(madElev, add = TRUE)

### Buffer a raster by a given number of cells:
buffByCells <- buffer(elev, width = 20.01, unit = "cells") # 20-cell buffer
plot(buffByCells, legend = FALSE)
plot(madElev, add = TRUE)

### Buffer a vector:
buffRivers <- buffer(rivers, width = 2000, dissolve = TRUE) # 2000-m buffer
plot(buffRivers)
plot(st_geometry(madRivers), col = "blue", add = TRUE)

}
}
\seealso{
\code{\link[terra:buffer]{terra::buffer()}}, \code{\link[sf:geos_unary]{sf::st_buffer()}}
}
