## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----setup--------------------------------------------------------------------
library(ethiodate)
# Parse a single date string
eth_parse_date("2015-01-01")

## -----------------------------------------------------------------------------
# Parse a vector of date strings
eth_dates <- c("2015-01-01", "2015-02-15", "2015-13-05")
eth_parse_date(eth_dates)

## -----------------------------------------------------------------------------
# Parsing dates in MM/DD/YYYY format
x <- c("01/17/2017", "05/12/2017")
eth_parse_date(x, format = "%m/%d/%Y")

## -----------------------------------------------------------------------------
# Using full English month names
eth_parse_date("Meskerem 25, 2017", format = "%B %d, %Y")

# Using full Amharic month names
eth_parse_date("መስከረም 25, 2017", format = "%B %d, %Y", lang = "amh")

# Using abbreviated English month names
eth_parse_date("Sep 25, 2017", format = "%b %d, %Y", lang = "en")

## -----------------------------------------------------------------------------
# Show the numeric representation of an Ethiopian date
eth_parse_date("1962-04-23") |> unclass()

# Show the numeric representation of the Gregorian origin
as.Date("1970-01-01") |> unclass()

## -----------------------------------------------------------------------------
# Create an ethdate object 7 days after today
eth_date(7, origin = eth_today())

# Create an ethdate object 7 days before today
eth_date(-7, origin = eth_today())

## -----------------------------------------------------------------------------
# Create a single date from components
eth_make_date(2017, 1, 1)

# Create multiple dates from component vectors
y <- c(2012, 2015)
m <- c(5, 9)
d <- c(15, 19)
eth_make_date(y, m, d)

## -----------------------------------------------------------------------------
# 2011 is a leap year in EC, so Pagume 6 is valid
eth_date("2011-13-06")

# 2012 is not a leap year in EC, so Pagume 6 is invalid
eth_date("2012-13-06")

# Attempting to create a date with month 14
eth_make_date(2016, 14, 1)

## -----------------------------------------------------------------------------
# Convert a Gregorian date to Ethiopian
gre_date <- as.Date("2025-01-15")
eth_date(gre_date)

## -----------------------------------------------------------------------------
# Convert an Ethiopian date (the origin) to Gregorian
eth_dt <- eth_date(0)
as.Date(eth_dt)

## -----------------------------------------------------------------------------
# Add 6 days to a date
eth_date("2010-09-14") + 6

# Subtract 6 days from a date
eth_date("2010-09-14") - 6

# Calculate the difference between two dates (in days)
eth_date("2010-09-14") - eth_date("2010-09-10")


## -----------------------------------------------------------------------------
ed1 <- eth_date("2010-09-14")
ed2 <- eth_date("2010-09-10")

ed1 > ed2

ed1 == ed2 + 4 


## -----------------------------------------------------------------------------
# Format today's date in a custom string
format(eth_today(), format = "This document was updated on %B %d, %Y EC.")

# Format with day of the week and year
format(eth_today(), format = "Today is %A, %B %d, %Y.")

# Format in Amharic
format(eth_today(), format = "%A، %B %d ቀን %Y ዓ.ም.", lang = "amh")

