% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Trials.R
\docType{class}
\name{Trials}
\alias{Trials}
\title{Class of Trial}
\description{
Create a class of trial.

Public methods in this R6 class are used in developing
this package. Thus, we have to export the whole R6 class which exposures all
public methods. However, only the public methods in the list below are
useful to end users.

\itemize{
\item \code{$set_duration()} set duration of a trial. This function can be
used to extend duration under adaptive designs.
\item \code{$remove_arms()} drop arms from a trial. This function can be
used in adaptive designs, e.g., dose selection, enrichment design, etc.
\item \code{$update_sample_ratio()} change sample ratio of arm. This function
can be used under adaptive designs, e.g., response-adaptive design, etc.
\item \code{$add_arms()} add arms to a trial. This function is used to add
arms to a newly defined trial, or add arms under adaptive design, e.g.,
dose-ranging, etc.
\item \code{$get_locked_data()} request for data snapshot at a milestone.
Calling this function is recommended as the first action in any action
function as long as trial data is needed in statistical analysis or decision
making.
\item \code{$save()} save intermediate result for simulation summary.
Results across multiple replicates of simulation are saved, which can be
retrieved by calling \code{get_output()} anytime.
\item \code{$bind()} row bind and save intermediate results across
milestones if those results are data frames of similar formats. The life
cycle of the save results is within a single replicate of simulation and
is reset to NULL in next simulated trial. Saved results
can be retrieved by calling \code{get()} anytime.
\item \code{$save_custom_data()} save intermediate results of any format.
The life cycle of the saved result is within a single replicate of simulation
and is reset to NULL in next simulated trial. Saved results can be retrieved
by calling \code{get()} anytime.
\item \code{$get()} retrieve intermediate results saved by calling functions
\code{save_custom_data()} or \code{bind()}.
\item \code{$get_output()} retrieve intermediate results saved by calling
function \code{save()}.
\item \code{$dunnettTest()} perform Dunnett's test.
\item \code{$closedTest()} perform combination test based on Dunnett's test.
}
}
\examples{
# Instead of using Trials$new, please use trial(), a user-friendly
# wrapper. See examples in ?trial.


## ------------------------------------------------
## Method `Trials$independentIncrement`
## ------------------------------------------------


\dontrun{
trial$independentIncrement(Surv(pfs, pfs_event) ~ arm, 'pbo',
                           listener$get_milestone_names(),
                           'less', 'oracle')
}

## ------------------------------------------------
## Method `Trials$dunnettTest`
## ------------------------------------------------

\dontrun{
trial$dunnettTest(Surv(pfs, pfs_event) ~ arm, 'pbo', c('high dose', 'low dose'),
                  listener$get_milestone_names(), 'default')
}


## ------------------------------------------------
## Method `Trials$closedTest`
## ------------------------------------------------

\dontrun{
dt <- trial$dunnettTest(
  Surv(pfs, pfs_event) ~ arm,
  placebo = 'pbo',
  treatments = c('high dose', 'low dose'),
  milestones = c('dose selection', 'interim', 'final'),
  data.frame(pbo = c(100, 160, 80),
             low = c(100, 160, 80),
             high = c(100, 160, 80),
             row.names = c('dose selection', 'interim', 'final'))

trial$closedTest(dt, treatments = c('high dose', 'low dose'),
                 milestones = c('interim', 'final'),
                 alpha = 0.025, alpha_spending = 'asOF')
}

}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-Trials-new}{\code{Trials$new()}}
\item \href{#method-Trials-get_trial_data}{\code{Trials$get_trial_data()}}
\item \href{#method-Trials-get_duration}{\code{Trials$get_duration()}}
\item \href{#method-Trials-set_duration}{\code{Trials$set_duration()}}
\item \href{#method-Trials-set_enroller}{\code{Trials$set_enroller()}}
\item \href{#method-Trials-get_enroller}{\code{Trials$get_enroller()}}
\item \href{#method-Trials-set_dropout}{\code{Trials$set_dropout()}}
\item \href{#method-Trials-get_dropout}{\code{Trials$get_dropout()}}
\item \href{#method-Trials-roll_back}{\code{Trials$roll_back()}}
\item \href{#method-Trials-remove_arms}{\code{Trials$remove_arms()}}
\item \href{#method-Trials-update_sample_ratio}{\code{Trials$update_sample_ratio()}}
\item \href{#method-Trials-add_arms}{\code{Trials$add_arms()}}
\item \href{#method-Trials-get_name}{\code{Trials$get_name()}}
\item \href{#method-Trials-get_description}{\code{Trials$get_description()}}
\item \href{#method-Trials-get_arms}{\code{Trials$get_arms()}}
\item \href{#method-Trials-get_arms_name}{\code{Trials$get_arms_name()}}
\item \href{#method-Trials-get_number_arms}{\code{Trials$get_number_arms()}}
\item \href{#method-Trials-has_arm}{\code{Trials$has_arm()}}
\item \href{#method-Trials-get_an_arm}{\code{Trials$get_an_arm()}}
\item \href{#method-Trials-get_sample_ratio}{\code{Trials$get_sample_ratio()}}
\item \href{#method-Trials-get_number_patients}{\code{Trials$get_number_patients()}}
\item \href{#method-Trials-get_number_enrolled_patients}{\code{Trials$get_number_enrolled_patients()}}
\item \href{#method-Trials-get_number_unenrolled_patients}{\code{Trials$get_number_unenrolled_patients()}}
\item \href{#method-Trials-get_randomization_queue}{\code{Trials$get_randomization_queue()}}
\item \href{#method-Trials-get_enroll_time}{\code{Trials$get_enroll_time()}}
\item \href{#method-Trials-enroll_patients}{\code{Trials$enroll_patients()}}
\item \href{#method-Trials-set_current_time}{\code{Trials$set_current_time()}}
\item \href{#method-Trials-get_current_time}{\code{Trials$get_current_time()}}
\item \href{#method-Trials-get_event_tables}{\code{Trials$get_event_tables()}}
\item \href{#method-Trials-get_data_lock_time_by_event_number}{\code{Trials$get_data_lock_time_by_event_number()}}
\item \href{#method-Trials-get_data_lock_time_by_calendar_time}{\code{Trials$get_data_lock_time_by_calendar_time()}}
\item \href{#method-Trials-get_locked_data}{\code{Trials$get_locked_data()}}
\item \href{#method-Trials-get_locked_data_name}{\code{Trials$get_locked_data_name()}}
\item \href{#method-Trials-get_event_number}{\code{Trials$get_event_number()}}
\item \href{#method-Trials-save_milestone_time}{\code{Trials$save_milestone_time()}}
\item \href{#method-Trials-get_milestone_time}{\code{Trials$get_milestone_time()}}
\item \href{#method-Trials-lock_data}{\code{Trials$lock_data()}}
\item \href{#method-Trials-event_plot}{\code{Trials$event_plot()}}
\item \href{#method-Trials-censor_trial_data}{\code{Trials$censor_trial_data()}}
\item \href{#method-Trials-save}{\code{Trials$save()}}
\item \href{#method-Trials-bind}{\code{Trials$bind()}}
\item \href{#method-Trials-save_custom_data}{\code{Trials$save_custom_data()}}
\item \href{#method-Trials-get_custom_data}{\code{Trials$get_custom_data()}}
\item \href{#method-Trials-get}{\code{Trials$get()}}
\item \href{#method-Trials-get_output}{\code{Trials$get_output()}}
\item \href{#method-Trials-mute}{\code{Trials$mute()}}
\item \href{#method-Trials-independentIncrement}{\code{Trials$independentIncrement()}}
\item \href{#method-Trials-dunnettTest}{\code{Trials$dunnettTest()}}
\item \href{#method-Trials-closedTest}{\code{Trials$closedTest()}}
\item \href{#method-Trials-get_seed}{\code{Trials$get_seed()}}
\item \href{#method-Trials-print}{\code{Trials$print()}}
\item \href{#method-Trials-get_snapshot_copy}{\code{Trials$get_snapshot_copy()}}
\item \href{#method-Trials-make_snapshot}{\code{Trials$make_snapshot()}}
\item \href{#method-Trials-make_arms_snapshot}{\code{Trials$make_arms_snapshot()}}
\item \href{#method-Trials-reset}{\code{Trials$reset()}}
\item \href{#method-Trials-set_arm_added_time}{\code{Trials$set_arm_added_time()}}
\item \href{#method-Trials-get_arm_added_time}{\code{Trials$get_arm_added_time()}}
\item \href{#method-Trials-set_arm_removal_time}{\code{Trials$set_arm_removal_time()}}
\item \href{#method-Trials-get_arm_removal_time}{\code{Trials$get_arm_removal_time()}}
\item \href{#method-Trials-clone}{\code{Trials$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-new"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-new}{}}}
\subsection{Method \code{new()}}{
initialize a trial
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$new(
  name,
  n_patients,
  duration,
  description = name,
  seed = NULL,
  enroller,
  dropout = NULL,
  silent = FALSE,
  ...
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{name}}{character. Name of trial. Usually, hmm..., useless.}

\item{\code{n_patients}}{integer. Maximum (and initial) number of patients
could be enrolled when planning the trial. It can be altered adaptively
during a trial.}

\item{\code{duration}}{Numeric. Trial duration. It can be altered adaptively
during a trial.}

\item{\code{description}}{character. Optional for description of the trial. By
default it is set to be trial's \code{name}. Usually useless.}

\item{\code{seed}}{random seed. If \code{NULL}, seed is set for each simulated
trial automatically and saved in output. It can be retrieved in the
\code{seed} column in \code{$get_output()}. Setting it to be \code{NULL}
is recommended. For debugging, set it to a specific integer.}

\item{\code{enroller}}{a function returning a vector enrollment time for
patients. Its first argument \code{n} is the number of enrolled patients.
Set it to \code{StaggeredRecruiter} can handle most of the use cases.
See \code{?TrialSimulator::StaggeredRecruiter} for more information.}

\item{\code{dropout}}{a function returning a vector of dropout time for patients.
It can be any random number generator with first argument \code{n},
the number of enrolled patients. Usually \code{rexp} if dropout rate
is set at a single time point, or \code{rweibull} if dropout rates are
set at two time points. See \code{?TrialSimulator::weibullDropout}.}

\item{\code{silent}}{logical. \code{TRUE} to mute messages. However, warning
message is still displayed.}

\item{\code{...}}{(optional) arguments of \code{enroller} and \code{dropout}.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_trial_data"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_trial_data}{}}}
\subsection{Method \code{get_trial_data()}}{
return trial data of enrolled patients at the time of this
function is called
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_trial_data()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_duration"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_duration}{}}}
\subsection{Method \code{get_duration()}}{
return maximum duration of a trial
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_duration()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-set_duration"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-set_duration}{}}}
\subsection{Method \code{set_duration()}}{
set trial duration in an adaptive designed trial. All patients enrolled
before resetting the duration are truncated (non-tte endpoints) or
censored (tte endpoints) at the original duration. Remaining patients
are re-randomized. New duration must be longer than the old one.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$set_duration(duration)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{duration}}{new duration of a trial. It must be greater than the
current duration.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-set_enroller"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-set_enroller}{}}}
\subsection{Method \code{set_enroller()}}{
set recruitment curve when initialize a trial.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$set_enroller(func, ...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{func}}{function to generate enrollment time. It can be built-in
function like `rexp` or customized functions like `StaggeredRecruiter`.}

\item{\code{...}}{(optional) arguments for \code{func}.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_enroller"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_enroller}{}}}
\subsection{Method \code{get_enroller()}}{
get function of recruitment curve
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_enroller()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-set_dropout"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-set_dropout}{}}}
\subsection{Method \code{set_dropout()}}{
set distribution of drop out time. This can be done when initialize a
trial, or when updating a trial in adaptive design.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$set_dropout(func, ...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{func}}{function to generate dropout time. It can be built-in
function like `rexp` or customized functions.}

\item{\code{...}}{(optional) arguments for \code{func}.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_dropout"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_dropout}{}}}
\subsection{Method \code{get_dropout()}}{
get generator of dropout time
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_dropout()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-roll_back"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-roll_back}{}}}
\subsection{Method \code{roll_back()}}{
roll back data to current time of trial. By doing so,
\code{Trial$trial_data} will be cut at current time, and data after then
are deleted. However, \code{Trial$enroll_time} after current time are
kept unchanged because that is planned enrollment curve.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$roll_back()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-remove_arms"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-remove_arms}{}}}
\subsection{Method \code{remove_arms()}}{
remove arms from a trial. \code{enroll_patients()} will be called
at the end of this function to enroll all remaining patients after
\code{Trials$get_current_time()}, i.e. no more unenrolled patients
could be randomized to removed arms. This function may be used with
futility analysis, dose selection, enrichment analysis (sub-population)
or interim analysis (early stop for efficacy).

Note that this function should only be called within action functions.
It is users' responsibility to ensure it and \code{TrialSimulator} has
no way to track this.
In addition, data of the removed arms are censored or truncated by
the time of arm removal.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$remove_arms(arms_name)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{arms_name}}{character vector. Name of arms to be removed.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-update_sample_ratio"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-update_sample_ratio}{}}}
\subsection{Method \code{update_sample_ratio()}}{
update sample ratios of arms. This could happen after an arm is added
or removed. Note that we may update sample ratios of unaffected arms as
well. Once sample ratio is updated, trial data should be rolled back with
updated randomization queue. Data of unenrolled patients are
re-sampled as well.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$update_sample_ratio(arm_names, sample_ratios)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{arm_names}}{character vector. Name of arms.}

\item{\code{sample_ratios}}{numeric vector. New sample ratios of arms. If
sample ratio is a whole number, the permuted block randomization is
adopted; otherwise, \code{sample()} will be used instead, which can
cause imbalance between arms by chance. However, this is fine for
simulation.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-add_arms"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-add_arms}{}}}
\subsection{Method \code{add_arms()}}{
add one or more arms to the trial. \code{enroll_patients()} will be
called at the end to enroll all remaining patients in
\code{private$randomization_queue}. This function can be used in two
scenarios:
(1) add arms right after a trial is created (i.e., \code{Trials$new(...)}).
\code{sample_ratio} and arms added through \code{...} should be of same
length;
(2) add arms to a trial already with arm(s).

Note that this function should only be called within action functions.
It is users' responsibility to ensure it and \code{TrialSimulator} has
no way to track this.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$add_arms(sample_ratio, ...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{sample_ratio}}{integer vector. Sample ratio for permuted block
randomization. It will be appended to existing sample ratio in the trial.}

\item{\code{...}}{one or more objects returned from \code{arm()}. One exception in
\code{...} is an argument \code{enforce}. When \code{enforce = TRUE},
sample ratio of newly added arm. It rolls back all patients after
\code{Trials$get_current_time()}, i.e. redo randomization for those
patients. This can be useful to add arms one by one when creating a trial.
Note that we can run \code{Trials$add_arm(sample_ratio1, arm1)} followed
by \code{Trials$add_arm(sample_ratio2, enforce = TRUE, arm2)}.
We would expected similar result with
\code{Trials$add_arms(c(sample_ratio1, sample_ratio2), arm1, arm2)}. Note
that these two method won't return exactly the same trial because
randomization_queue were generated twice in the first approach but only
once in the second approach. But statistically, they are equivalent and
of the same distribution.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_name"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_name}{}}}
\subsection{Method \code{get_name()}}{
return name of trial
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_name()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_description"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_description}{}}}
\subsection{Method \code{get_description()}}{
return description of trial
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_description()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_arms"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_arms}{}}}
\subsection{Method \code{get_arms()}}{
return a list of arms in the trial
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_arms()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_arms_name"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_arms_name}{}}}
\subsection{Method \code{get_arms_name()}}{
return arms' name of trial
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_arms_name()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_number_arms"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_number_arms}{}}}
\subsection{Method \code{get_number_arms()}}{
get number of arms in the trial
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_number_arms()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-has_arm"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-has_arm}{}}}
\subsection{Method \code{has_arm()}}{
check if the trial has any arm. Return \code{TRUE} or \code{FALSE}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$has_arm()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_an_arm"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_an_arm}{}}}
\subsection{Method \code{get_an_arm()}}{
return an arm
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_an_arm(arm_name)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{arm_name}}{character, name of arm to be extracted}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_sample_ratio"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_sample_ratio}{}}}
\subsection{Method \code{get_sample_ratio()}}{
return current sample ratio of the trial. The ratio can probably change
during the trial (e.g., arm is removed or added)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_sample_ratio(arm_names = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{arm_names}}{character vector of arms.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_number_patients"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_number_patients}{}}}
\subsection{Method \code{get_number_patients()}}{
return number of patients when planning the trial
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_number_patients()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_number_enrolled_patients"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_number_enrolled_patients}{}}}
\subsection{Method \code{get_number_enrolled_patients()}}{
return number of enrolled (randomized) patients
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_number_enrolled_patients()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_number_unenrolled_patients"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_number_unenrolled_patients}{}}}
\subsection{Method \code{get_number_unenrolled_patients()}}{
return number of unenrolled patients
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_number_unenrolled_patients()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_randomization_queue"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_randomization_queue}{}}}
\subsection{Method \code{get_randomization_queue()}}{
return randomization queue of planned but not yet enrolled patients.
This function does not update randomization_queue, just return its value
for debugging purpose.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_randomization_queue(index = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index}}{index to be extracted. Return all queue if \code{NULL}.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_enroll_time"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_enroll_time}{}}}
\subsection{Method \code{get_enroll_time()}}{
return enrollment time of planned but not yet enrolled patients.
This function does not update enroll_time, just return its value
for debugging purpose.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_enroll_time(index = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index}}{index to extract. Return all enroll time if \code{NULL}.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-enroll_patients"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-enroll_patients}{}}}
\subsection{Method \code{enroll_patients()}}{
assign new patients to pre-planned randomization queue at pre-specified
enrollment time.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$enroll_patients(n_patients = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{n_patients}}{number of new patients to be enrolled. If \code{NULL},
all remaining patients in plan are enrolled. Error may be triggered if
n_patients is greater than remaining patients as planned.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-set_current_time"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-set_current_time}{}}}
\subsection{Method \code{set_current_time()}}{
set current time of a trial. Any data collected before could not be
changed. private$now should be set after a milestone is triggered
(through Milestones class, futility, interim, etc), an arm is added or
removed at a milestone
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$set_current_time(time)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{time}}{current calendar time of a trial.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_current_time"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_current_time}{}}}
\subsection{Method \code{get_current_time()}}{
return current time of a trial
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_current_time()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_event_tables"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_event_tables}{}}}
\subsection{Method \code{get_event_tables()}}{
count accumulative number of events (for TTE) or non-missing samples (otherwise) over
calendar time (enroll time + tte for TTE, or enroll time + readout otherwise)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_event_tables(arms = NULL, ...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{arms}}{a vector of arms' name on which the event tables are created.
if \code{NULL}, all arms in the trial will be used.}

\item{\code{...}}{subset conditions compatible with \code{dplyr::filter}.
Event tables will be counted on subset of trial data only.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_data_lock_time_by_event_number"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_data_lock_time_by_event_number}{}}}
\subsection{Method \code{get_data_lock_time_by_event_number()}}{
given a set of endpoints and target number of events, determine the data
lock time for a milestone (futility, interim, final, etc.). This function does
not change trial object (e.g. rolling back not yet randomized patients after
the found data lock time).
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_data_lock_time_by_event_number(
  endpoints,
  arms,
  target_n_events,
  type = c("all", "any"),
  ...
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{endpoints}}{character vector. Data lock time is determined by a set
of endpoints.}

\item{\code{arms}}{a vector of arms' name on which number of events will be
counted.}

\item{\code{target_n_events}}{target number of events for each of the
\code{endpoints}.}

\item{\code{type}}{\code{all} if all target number of events are reached.
\code{any} if the any target number of events is reached.}

\item{\code{...}}{subset conditions compatible with \code{dplyr::filter}. Number
Time of milestone is based on event counts on the subset of trial data.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
data lock time
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_data_lock_time_by_calendar_time"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_data_lock_time_by_calendar_time}{}}}
\subsection{Method \code{get_data_lock_time_by_calendar_time()}}{
given the calendar time to lock the data, return it with event counts of
each of the endpoints.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_data_lock_time_by_calendar_time(calendar_time, arms)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{calendar_time}}{numeric. Calendar time to lock the data}

\item{\code{arms}}{a vector of arms' name on which number of events will be
counted.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
data lock time
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_locked_data"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_locked_data}{}}}
\subsection{Method \code{get_locked_data()}}{
return locked data, i.e. snapshot at a milestone. TTE data is censored
and non-TTE data is truncated accounting for readout time and dropout
time simultaneously by the triggering time of milestone.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_locked_data(milestone_name)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{milestone_name}}{character. Milestone name of which the locked
data to be extracted.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_locked_data_name"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_locked_data_name}{}}}
\subsection{Method \code{get_locked_data_name()}}{
return names of locked data
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_locked_data_name()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_event_number"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_event_number}{}}}
\subsection{Method \code{get_event_number()}}{
return number of events at lock time of milestones
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_event_number(milestone_name = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{milestone_name}}{names of triggered milestones. Use all triggered milestones
if \code{NULL}.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-save_milestone_time"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-save_milestone_time}{}}}
\subsection{Method \code{save_milestone_time()}}{
save time of a new milestone.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$save_milestone_time(milestone_time, milestone_name)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{milestone_time}}{numeric. Time of new milestone.}

\item{\code{milestone_name}}{character. Name of new milestone.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_milestone_time"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_milestone_time}{}}}
\subsection{Method \code{get_milestone_time()}}{
return milestone time when triggering a given milestone
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_milestone_time(milestone_name = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{milestone_name}}{character. Name of milestone. If \code{NULL},
time of all triggered milestones are returned.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-lock_data"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-lock_data}{}}}
\subsection{Method \code{lock_data()}}{
lock data at specific calendar time.
For time-to-event endpoints, their event indicator \code{*_event} should be
updated accordingly. Locked data should be stored separately.
DO NOT OVERWRITE/UPDATE private$trial_data! which can lose actual
time-to-event information. For example, a patient may be censored at
the first data lock. However, he may have event being observed in a
later data lock.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$lock_data(at_calendar_time, milestone_name)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{at_calendar_time}}{time point to lock trial data}

\item{\code{milestone_name}}{assign milestone name as the name of locked data for
future reference.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-event_plot"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-event_plot}{}}}
\subsection{Method \code{event_plot()}}{
plot of cumulative number of events/samples over calendar time.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$event_plot()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-censor_trial_data"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-censor_trial_data}{}}}
\subsection{Method \code{censor_trial_data()}}{
censor trial data at calendar time
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$censor_trial_data(
  censor_at = NULL,
  selected_arms = NULL,
  enrolled_before = Inf
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{censor_at}}{time of censoring. It is set to trial duration if
\code{NULL}.}

\item{\code{selected_arms}}{censoring is applied to selected arms (e.g.,
removed arms) only. If \code{NULL}, it will be set to all available arms
in trial data. Otherwise, censoring is applied to user-specified arms only.
This is necessary because number of events/sample size in removed arms
should be fixed unchanged since corresponding milestone is triggered. In that
case, one can update trial data by something like
\code{censor_trial_data(censor_at = milestone_time, selected_arms = removed_arms)}.}

\item{\code{enrolled_before}}{censoring is applied to patients enrolled before
specific time. This argument would be used when trial duration is
updated by \code{set_duration}. Adaptation happens when \code{set_duration}
is called so we fix duration for patients enrolled before adaptation
to maintain independent increment. This should work when trial duration
is updated for multiple times.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-save"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-save}{}}}
\subsection{Method \code{save()}}{
save a single value or a one-row data frame to trial's output
for further analysis/summary later. Results saved by calling this
function have a life cycle of the whole simulation. This means that
all results are accumulated across multiple simulated trial and can be
used for summary later.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$save(value, name = "", overwrite = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{value}}{value to be saved. It can be a scalar (vector of length 1)
or a data frame (of one row).}

\item{\code{name}}{character to name the saved object. It will be used to
name a column in trial's output if \code{value} is a scalar.
If \code{value} is a data frame, \code{name} will be the prefix pasted
with the column name of \code{value} in trial's output.
If user want to use
\code{value}'s column name as is in trial's output, set \code{name}
to be \code{''} as default. Otherwise, column name would be, e.g.,
\code{"{<name>}_<{colnames(value)}>"}.}

\item{\code{overwrite}}{logic. \code{TRUE} if overwriting existing entries
with warning, otherwise, throwing an error and stop.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-bind"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-bind}{}}}
\subsection{Method \code{bind()}}{
row bind a data frame to existing data frame. If a data frame \code{name}
is not existing in a trial, then it is equivalent to
calling \code{Trials$save_custom_data()}.
Extra columns in \code{value} are ignored. Columns in
\code{Trials$custom_data[[name]]} but not in \code{value} are filled
with \code{NA}.

This function can be used to save results across multiple milestones.
For example, p-values and effect estimates of endpoints may be computed
at multiple milestones. Users may want to bind them into a data frame
for combination test or graphical test. In this case, this function
can be called repeatedly in milestones. Once the data frame is fully
conducted, statistical test can be performed on its final version
retrieved by calling \code{Trials$get()}.

Note that data saved by calling this function has a short life cycle
within a single simulated trial. It will be reset to \code{NULL} before
simulated another trial. Thus, it cannot be used to save results that
are used for summarizing the simulation.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$bind(value, name)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{value}}{a data frame to be saved. It can consist of one or
multiple rows.}

\item{\code{name}}{character. Name of object to be saved.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-save_custom_data"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-save_custom_data}{}}}
\subsection{Method \code{save_custom_data()}}{
save arbitrary (number of) objects into a trial so that users can use
those to control the workflow. Most common use case is to store
simulation parameters to be used in action functions.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$save_custom_data(value, name, overwrite = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{value}}{value to be saved. Any type.}

\item{\code{name}}{character. Name of the value to be accessed later.}

\item{\code{overwrite}}{logic. \code{TRUE} if overwriting existing entries
with warning, otherwise, throwing an error and stop.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_custom_data"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_custom_data}{}}}
\subsection{Method \code{get_custom_data()}}{
return custom data saved by calling \code{Trials$save_custom_data()}
or \code{Trials$bind()} with its name.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_custom_data(name)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{name}}{character. Name of custom data to be accessed.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get}{}}}
\subsection{Method \code{get()}}{
alias of function \code{get_custom_data} to make it short and cool.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get(name)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{name}}{character. Name of custom data to be accessed.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_output"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_output}{}}}
\subsection{Method \code{get_output()}}{
return a data frame of all current outputs saved by calling
\code{Trials$save()}. Usually this function is call at the end of
simulation for summary.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_output(cols = NULL, simplify = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{cols}}{columns to be returned from \code{Trial$output}. If
\code{NULL}, all columns are returned.}

\item{\code{simplify}}{logical. Return value rather than a data frame of one
column when \code{length(col) == 1} and \code{simplify == TRUE}.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-mute"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-mute}{}}}
\subsection{Method \code{mute()}}{
mute all messages (not including warnings)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$mute(silent)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{silent}}{logical.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-independentIncrement"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-independentIncrement}{}}}
\subsection{Method \code{independentIncrement()}}{
calculate independent increments from a given set of milestones
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$independentIncrement(
  formula,
  placebo,
  milestones,
  alternative,
  planned_info,
  ...
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{formula}}{An object of class \code{formula} that can be used with
\code{survival::coxph}. Must consist \code{arm} and endpoint in \code{data}.
No covariate is allowed. Stratification variables are supported and can be
added using \code{strata(...)}.}

\item{\code{placebo}}{character. String of placebo in trial's locked data.}

\item{\code{milestones}}{a character vector of milestone names in the trial, e.g.,
\code{listener$get_milestone_names()}.}

\item{\code{alternative}}{a character string specifying the alternative hypothesis,
must be one of \code{"greater"} or \code{"less"}. No default value.
\code{"greater"} means superiority of treatment over placebo is established
by an hazard ratio greater than 1 when a log-rank test is used.}

\item{\code{planned_info}}{a vector of planned accumulative number of event of
time-to-event endpoint. It is named by milestone names.
Note: \code{planned_info} can also be a character
\code{"oracle"} so that planned number of events are set to be observed
number of events, in that case inverse normal z statistics equal to
one-sided logrank statistics. This is for the purpose of debugging only.
In formal simulation, \code{"oracle"} should not be used if adaptation
is present. Pre-fixed \code{planned_info} should be used to create
weights in combination test that controls the family-wise error rate
in the strong sense.}

\item{\code{...}}{subset condition that is compatible with \code{dplyr::filter}.
\code{survdiff} will be fitted on this subset only to compute one-sided
logrank statistics. It could be useful when a
trial consists of more than two arms. By default it is not specified,
all data will be used to fit the model.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
This function returns a data frame with columns:
\describe{
\item{\code{p_inverse_normal}}{one-sided p-value for inverse normal test
based on logrank test (alternative hypothesis: risk is higher in placebo arm).
Accumulative data is used. }
\item{\code{z_inverse_normal}}{z statistics of \code{p_inverse_normal}.
Accumulative data is used. }
\item{\code{p_lr}}{one-sided p-value for logrank test
 (alternative hypothesis: risk is higher in placebo arm).
Accumulative data is used. }
\item{\code{z_lr}}{z statistics of \code{p_lr}.
Accumulative data is used. }
\item{\code{info}}{observed accumulative event number. }
\item{\code{planned_info}}{planned accumulative event number. }
\item{\code{info_pbo}}{observed accumulative event number in placebo. }
\item{\code{info_trt}}{observed accumulative event number in treatment arm. }
\item{\code{wt}}{weights in \code{z_inverse_normal}. }
}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{
\dontrun{
trial$independentIncrement(Surv(pfs, pfs_event) ~ arm, 'pbo',
                           listener$get_milestone_names(),
                           'less', 'oracle')
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-dunnettTest"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-dunnettTest}{}}}
\subsection{Method \code{dunnettTest()}}{
carry out closed test based on Dunnett method under group sequential
design.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$dunnettTest(
  formula,
  placebo,
  treatments,
  milestones,
  alternative,
  planned_info,
  ...
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{formula}}{An object of class \code{formula} that can be used with
\code{survival::coxph}. Must consist \code{arm} and endpoint in \code{data}.
No covariate is allowed. Stratification variables are supported and can be
added using \code{strata(...)}.}

\item{\code{placebo}}{character. Name of placebo arm.}

\item{\code{treatments}}{character vector. Name of treatment arms to be used in
comparison.}

\item{\code{milestones}}{character vector. Names of triggered milestones at which either
adaptation is applied or statistical testing for endpoint is performed.
Milestones in \code{milestones} does not need to be sorted by their triggering time.}

\item{\code{alternative}}{a character string specifying the alternative hypothesis,
must be one of \code{"greater"} or \code{"less"}. No default value.
\code{"greater"} means superiority of treatment over placebo is established
by an hazard ratio greater than 1 when a log-rank test is used.}

\item{\code{planned_info}}{a data frame of planned number of events of
time-to-event endpoint in each stage and each arm. Milestone names, i.e.,
\code{milestones} are row names of \code{planned_info}, and arm names, i.e.,
\code{c(placebo, treatments)} are column names.
Note that it is not the accumulative but stage-wise event numbers.
It is usually not easy to determine these numbers in practice, simulation
may be used to get estimates.
Note: \code{planned_info} can also be a character
\code{"default"} so that \code{planned_info} are set to be number
of newly randomized patients in the control arm in each of the stages.
This assumes that
event rate do not change over time and/or sample ratio between placebo
and a treatment arm does not change as well, which may not be true.
It is for the purpose of debugging or rapid implementation
only. Using simulation to pick \code{planned_info} is recommended in
formal simulation study. Another issue with \code{planned_info} set to
be \code{"default"} is that it is possible patient recruitment is done
before a specific stage, as a result, \code{planned_info} is zero which
can crash the program.}

\item{\code{...}}{subset condition that is compatible with \code{dplyr::filter}.
\code{survdiff} will be fitted on this subset only to compute one-sided
logrank statistics. It could be useful when comparison is made on a
subset of treatment arms. By default it is not specified,
all data (placebo plus one treatment arm at a time) in the locked data
are used to fit the model.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
This function computes stage-wise p-values for each of the intersection
hypotheses based on Dunnett test. If only one treatment arm is present,
it is equivalent to compute the stage-wise p-values of elemental
hypotheses. This function also computes inverse normal combination
test statistics at each of the stages.
The choice of \code{planned_info} can affect the calculation of
stage-wise p-values. Specifically, it is used to compute
the columns \code{observed_info} and \code{p_inverse_normal} in returned
data frame, which will be used in \code{Trial$closedTest()}.
The choice of \code{planned_info} can affect the result of
\code{Trial$closedTest()} so user should chose it with caution.

Note that in \code{Trial$closedTest()},
\code{observed_info}, which is derived from \code{planned_info}, will
lead to the same closed testing results up to a constant. This is because
the closed test uses information fraction
\code{observed_info/sum(observed_info)}. As a result, setting
\code{planned_info} to, e.g., \code{10 * planned_info} should give same
closed test results.

Based on numerical study, setting \code{planned_info = "default"} leads
to a much higher power (roughly 10\%) than setting \code{planned_info} to
median of event numbers at stages, which can be determined by simulation.
I am not sure if regulator would support such practice. For example,
if a milestone (e.g., interim analysis) is triggered at a pre-specified
calendar time, the number of randomized patients is random and is unknown
when planning the trial. If I understand it correctly, regulator may want
the information fraction in closed test (combined with Dunnett test) to
be pre-fixed. In addition, this choice for \code{planned_info} assumes
that the event rates does not change over time which is obviously not
true. It is recommended to always use pre-fixed \code{planned_info} for
restrict control of family-wise error rate. It should be pointed out
that the choice of pre-fixed \code{planned_info} can affect statistical
power significantly so fine-tuning may be required.
}

\subsection{Returns}{
a list with element names like \code{arm_name},
\code{arm1_name|arm2_name}, \code{arm1_name|arm2_name|arm3_name}, etc.,
i.e., all possible combination of treatment arms in comparison. Each
element is a data frame, with its column names self-explained. Specifically,
the columns \code{p_inverse_normal}, \code{observed_info},
\code{is_final} can be used with \code{GroupSequentialTest} to perform
significance test.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
trial$dunnettTest(Surv(pfs, pfs_event) ~ arm, 'pbo', c('high dose', 'low dose'),
                  listener$get_milestone_names(), 'default')
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-closedTest"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-closedTest}{}}}
\subsection{Method \code{closedTest()}}{
perform closed test based on Dunnett test
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$closedTest(
  dunnett_test,
  treatments,
  milestones,
  alpha,
  alpha_spending = c("asP", "asOF")
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{dunnett_test}}{object returned by \code{Trial$dunnettTest()}.}

\item{\code{treatments}}{character vector. Name of treatment arms to be used in
comparison.}

\item{\code{milestones}}{character vector. Names of triggered milestones at which
significance testing for endpoint is performed in closed test.
Milestones in \code{milestones} does not need to be sorted by their triggering time.}

\item{\code{alpha}}{numeric. Allocated alpha.}

\item{\code{alpha_spending}}{alpha spending function. It can be \code{"asP"} or
\code{"asOF"}. Note that theoretically it can be \code{"asUser"}, but
it is not tested. It may be supported in the future.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
a data frame of columns \code{arm}, \code{decision}
(final decision on a hypothesis at the end of trial, \code{"accept"} or \code{"reject"}),
\code{milestone_at_reject}, and \code{reject_time}.
If a hypothesis is accepted at then end of a trial,
\code{milestone_at_reject} is \code{NA}, and \code{reject_time} is \code{Inf}.

Note that if a hypothesis is tested at multiple milestones, the final
\code{decision} will be \code{"accept"} if it is accepted at at least
one milestone. The \code{decision} is \code{"reject"} only if the hypothesis
is rejected at all milestones.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
dt <- trial$dunnettTest(
  Surv(pfs, pfs_event) ~ arm,
  placebo = 'pbo',
  treatments = c('high dose', 'low dose'),
  milestones = c('dose selection', 'interim', 'final'),
  data.frame(pbo = c(100, 160, 80),
             low = c(100, 160, 80),
             high = c(100, 160, 80),
             row.names = c('dose selection', 'interim', 'final'))

trial$closedTest(dt, treatments = c('high dose', 'low dose'),
                 milestones = c('interim', 'final'),
                 alpha = 0.025, alpha_spending = 'asOF')
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_seed"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_seed}{}}}
\subsection{Method \code{get_seed()}}{
return random seed
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_seed()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-print"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-print}{}}}
\subsection{Method \code{print()}}{
print a trial
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$print()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_snapshot_copy"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_snapshot_copy}{}}}
\subsection{Method \code{get_snapshot_copy()}}{
return a snapshot of a trial before it is executed.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_snapshot_copy()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-make_snapshot"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-make_snapshot}{}}}
\subsection{Method \code{make_snapshot()}}{
make a snapshot before running a trial. This can be useful when
resetting a trial. This is only called when initializing a `Trial`
object, when arms have not been added yet.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$make_snapshot()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-make_arms_snapshot"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-make_arms_snapshot}{}}}
\subsection{Method \code{make_arms_snapshot()}}{
make a snapshot of arms
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$make_arms_snapshot()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-reset"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-reset}{}}}
\subsection{Method \code{reset()}}{
reset a trial to its snapshot taken before it was executed. Seed will be
reassigned with a new one. Enrollment time are re-generated. If the trial
already have arms when this function is called, they are added back to
recruit patients again.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$reset()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-set_arm_added_time"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-set_arm_added_time}{}}}
\subsection{Method \code{set_arm_added_time()}}{
save time when an arm is added to the trial
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$set_arm_added_time(arm, time)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{arm}}{name of added arm.}

\item{\code{time}}{time when an arm is added.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_arm_added_time"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_arm_added_time}{}}}
\subsection{Method \code{get_arm_added_time()}}{
get time when an arm is added to the trial
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_arm_added_time(arm)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{arm}}{arm name.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-set_arm_removal_time"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-set_arm_removal_time}{}}}
\subsection{Method \code{set_arm_removal_time()}}{
save time when an arm is removed to the trial
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$set_arm_removal_time(arm, time)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{arm}}{name of removed arm.}

\item{\code{time}}{time when an arm is removed.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-get_arm_removal_time"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-get_arm_removal_time}{}}}
\subsection{Method \code{get_arm_removal_time()}}{
get time when an arm is removed from the trial
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$get_arm_removal_time(arm)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{arm}}{arm name.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Trials-clone"></a>}}
\if{latex}{\out{\hypertarget{method-Trials-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Trials$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
