# =============================================================================
# EM ALGORITHM WITH GIBBS M-STEP (RECURSIVE ECF)
# =============================================================================


#' EM algorithm for alpha-stable mixture using recursive ECF and Gibbs M-step
#'
#' Performs EM estimation of a two-component alpha-stable mixture using recursive
#' kernel smoothing on the empirical characteristic function (ECF), with the M-step
#' replaced by Gibbs sampling.
#'
#' @param data Numeric vector of observations.
#' @param max_iter Maximum number of EM iterations.
#' @param tol Convergence tolerance on log-likelihood.
#' @return A list with estimated component parameters (\code{alpha}, \code{beta}, \code{gamma}, \code{delta}) and mixture weight (\code{w}).
#' @importFrom stats kmeans
#' @export
em_estimate_stable_recursive_ecf_with_gibbs <- function(data, max_iter = 100, tol = 1e-4) {
  u <- seq(0.1, 1, length.out = 10)
  labels <- kmeans(matrix(data, ncol = 1), centers = 2)$cluster
  w <- mean(labels == 1)

  params1 <- if (sum(labels == 1) < 5) {
    as.list(setNames(stable_fit_init(data), c("alpha", "beta", "gamma", "delta")))
  } else {
    estimate_stable_recursive_ecf(data[labels == 1], u)
  }

  params2 <- if (sum(labels == 2) < 5) {
    as.list(setNames(stable_fit_init(data), c("alpha", "beta", "gamma", "delta")))
  } else {
    estimate_stable_recursive_ecf(data[labels == 2], u)
  }

  log_likelihood_old <- -Inf

  for (iteration in 1:max_iter) {
    pdf1 <- pmax(r_stable_pdf(data, params1$alpha, params1$beta, params1$gamma, params1$delta), 1e-300)
    pdf2 <- pmax(r_stable_pdf(data, params2$alpha, params2$beta, params2$gamma, params2$delta), 1e-300)

    resp1 <- w * pdf1
    resp2 <- (1 - w) * pdf2
    total <- resp1 + resp2

    gamma1 <- resp1 / total
    gamma2 <- resp2 / total
    labels <- ifelse(gamma1 > gamma2, 1, 2)
    w <- mean(labels == 1)

    if (sum(labels == 1) >= 5) {
      best1 <- mock_gibbs_sampling(data[labels == 1])[[1]]
      params1 <- list(alpha = best1[2], beta = best1[3], gamma = best1[4], delta = best1[5])
    } else {
      message("Cluster 0 too small. Reusing previous estimate.\n")
    }

    if (sum(labels == 2) >= 5) {
      best2 <- mock_gibbs_sampling(data[labels == 2])[[1]]
      params2 <- list(alpha = best2[6], beta = best2[7], gamma = best2[8], delta = best2[9])
    } else {
      message("Cluster 1 too small. Reusing previous estimate.\n")
    }

    pdf1 <- pmax(r_stable_pdf(data, params1$alpha, params1$beta, params1$gamma, params1$delta), 1e-300)
    pdf2 <- pmax(r_stable_pdf(data, params2$alpha, params2$beta, params2$gamma, params2$delta), 1e-300)
    log_likelihood <- sum(log(w * pdf1 + (1 - w) * pdf2))

    message(sprintf("[Gibbs EM] Iteration %d: Log-Likelihood = %.6f\n", iteration, log_likelihood))

    if (abs(log_likelihood - log_likelihood_old) < tol) {
      message(sprintf("converged after %d iterations.\n", iteration))
      break
    }

    log_likelihood_old <- log_likelihood
  }

  return(list(params1 = params1, params2 = params2, weight = w))
}


# =============================================================================
# EM ALGORITHM WITH GIBBS M-STEP (KERNEL ECF)
# =============================================================================


#' EM algorithm for alpha-stable mixture using kernel ECF and Gibbs M-step
#'
#' Performs EM estimation of a two-component alpha-stable mixture using kernel-smoothed
#' empirical characteristic function (ECF), with the M-step replaced by Gibbs sampling.
#'
#' @param data Numeric vector of observations.
#' @param max_iter Maximum number of EM iterations.
#' @param tol Convergence tolerance on log-likelihood.
#' @return A list with estimated component parameters (\code{alpha}, \code{beta}, \code{gamma}, \code{delta}) and mixture weight (\code{w}).
#' @importFrom stats kmeans
#' @export
em_estimate_stable_kernel_ecf_with_gibbs <- function(data, max_iter = 100, tol = 1e-4) {
  u <- seq(0.1, 1, length.out = 10)
  labels <- kmeans(matrix(data, ncol = 1), centers = 2)$cluster
  w <- mean(labels == 1)

  params1 <- if (sum(labels == 1) < 5) {
    as.list(setNames(stable_fit_init(data), c("alpha", "beta", "gamma", "delta")))
  } else {
    estimate_stable_kernel_ecf(data[labels == 1], u)
  }

  params2 <- if (sum(labels == 2) < 5) {
    as.list(setNames(stable_fit_init(data), c("alpha", "beta", "gamma", "delta")))
  } else {
    estimate_stable_kernel_ecf(data[labels == 2], u)
  }

  log_likelihood_old <- -Inf

  for (iteration in 1:max_iter) {
    pdf1 <- pmax(r_stable_pdf(data, params1$alpha, params1$beta, params1$gamma, params1$delta), 1e-300)
    pdf2 <- pmax(r_stable_pdf(data, params2$alpha, params2$beta, params2$gamma, params2$delta), 1e-300)

    resp1 <- w * pdf1
    resp2 <- (1 - w) * pdf2
    total <- resp1 + resp2

    gamma1 <- resp1 / total
    gamma2 <- resp2 / total
    labels <- ifelse(gamma1 > gamma2, 1, 2)
    w <- mean(labels == 1)

    if (sum(labels == 1) >= 5) {
      best1 <- mock_gibbs_sampling(data[labels == 1])[[1]]
      params1 <- list(alpha = best1[2], beta = best1[3], gamma = best1[4], delta = best1[5])
    } else {
      message("Cluster 0 too small. Reusing previous estimate.\n")
    }

    if (sum(labels == 2) >= 5) {
      best2 <- mock_gibbs_sampling(data[labels == 2])[[1]]
      params2 <- list(alpha = best2[6], beta = best2[7], gamma = best2[8], delta = best2[9])
    } else {
      message("Cluster 1 too small. Reusing previous estimate.\n")
    }

    pdf1 <- pmax(r_stable_pdf(data, params1$alpha, params1$beta, params1$gamma, params1$delta), 1e-300)
    pdf2 <- pmax(r_stable_pdf(data, params2$alpha, params2$beta, params2$gamma, params2$delta), 1e-300)
    log_likelihood <- sum(log(w * pdf1 + (1 - w) * pdf2))

    message(sprintf("[Gibbs EM] Iteration %d: Log-Likelihood = %.6f\n", iteration, log_likelihood))

    if (abs(log_likelihood - log_likelihood_old) < tol) {
      message(sprintf("converged after %d iterations.\n", iteration))
      break
    }

    log_likelihood_old <- log_likelihood
  }

  return(list(params1 = params1, params2 = params2, weight = w))
}


# =============================================================================
# EM ALGORITHM WITH GIBBS M-STEP (WEIGHTED OLS)
# =============================================================================


#' EM algorithm for alpha-stable mixture using weighted OLS and Gibbs M-step
#'
#' Performs EM estimation of a two-component alpha-stable mixture using weighted
#' least squares regression on the ECF, with the M-step replaced by Gibbs sampling.
#'
#' @param data Numeric vector of observations.
#' @param max_iter Maximum number of EM iterations.
#' @param tol Convergence tolerance on log-likelihood.
#' @return A list with estimated component parameters (\code{alpha}, \code{beta}, \code{gamma}, \code{delta}) and mixture weight (\code{w}).
#' @importFrom stats kmeans
#' @export
em_estimate_stable_weighted_ols_with_gibbs <- function(data, max_iter = 100, tol = 1e-4) {
  u <- seq(0.1, 1, length.out = 10)
  labels <- kmeans(matrix(data, ncol = 1), centers = 2)$cluster
  w <- mean(labels == 1)

  params1 <- if (sum(labels == 1) < 5) {
    as.list(setNames(stable_fit_init(data), c("alpha", "beta", "gamma", "delta")))
  } else {
    estimate_stable_weighted_ols(data[labels == 1], u)
  }

  params2 <- if (sum(labels == 2) < 5) {
    as.list(setNames(stable_fit_init(data), c("alpha", "beta", "gamma", "delta")))
  } else {
    estimate_stable_weighted_ols(data[labels == 2], u)
  }

  log_likelihood_old <- -Inf

  for (iteration in 1:max_iter) {
    pdf1 <- pmax(r_stable_pdf(data, params1$alpha, params1$beta, params1$gamma, params1$delta), 1e-300)
    pdf2 <- pmax(r_stable_pdf(data, params2$alpha, params2$beta, params2$gamma, params2$delta), 1e-300)

    resp1 <- w * pdf1
    resp2 <- (1 - w) * pdf2
    total <- resp1 + resp2

    gamma1 <- resp1 / total
    gamma2 <- resp2 / total
    labels <- ifelse(gamma1 > gamma2, 1, 2)
    w <- mean(labels == 1)

    if (sum(labels == 1) >= 5) {
      best1 <- mock_gibbs_sampling(data[labels == 1])[[1]]
      params1 <- list(alpha = best1[2], beta = best1[3], gamma = best1[4], delta = best1[5])
    } else {
      message("Cluster 0 too small. Reusing previous estimate.\n")
    }

    if (sum(labels == 2) >= 5) {
      best2 <- mock_gibbs_sampling(data[labels == 2])[[1]]
      params2 <- list(alpha = best2[6], beta = best2[7], gamma = best2[8], delta = best2[9])
    } else {
      message("Cluster 1 too small. Reusing previous estimate.\n")
    }

    pdf1 <- pmax(r_stable_pdf(data, params1$alpha, params1$beta, params1$gamma, params1$delta), 1e-300)
    pdf2 <- pmax(r_stable_pdf(data, params2$alpha, params2$beta, params2$gamma, params2$delta), 1e-300)
    log_likelihood <- sum(log(w * pdf1 + (1 - w) * pdf2))

    message(sprintf("[Gibbs EM] Iteration %d: Log-Likelihood = %.6f\n", iteration, log_likelihood))

    if (abs(log_likelihood - log_likelihood_old) < tol) {
      message(sprintf("converged after %d iterations.\n", iteration))
      break
    }

    log_likelihood_old <- log_likelihood
  }

  return(list(params1 = params1, params2 = params2, weight = w))
}


# =============================================================================
# EM ALGORITHM WITH GIBBS M-STEP (CDF-BASED ESTIMATION)
# =============================================================================


#' EM algorithm for alpha-stable mixture using CDF-based ECF and Gibbs M-step
#'
#' Performs EM estimation of a two-component alpha-stable mixture using a simplified
#' empirical characteristic function derived from the cumulative distribution function (CDF),
#' with the M-step replaced by Gibbs sampling.
#'
#' @param data Numeric vector of observations.
#' @param max_iter Maximum number of EM iterations.
#' @param tol Convergence tolerance on log-likelihood.
#' @return A list with estimated component parameters (\code{alpha}, \code{beta}, \code{gamma}, \code{delta}) and mixture weight (\code{w}).
#' @importFrom stats kmeans
#' @export
em_estimate_stable_from_cdf_with_gibbs <- function(data, max_iter = 100, tol = 1e-4) {
  u <- seq(0.1, 1, length.out = 10)
  labels <- kmeans(matrix(data, ncol = 1), centers = 2)$cluster
  w <- mean(labels == 1)

  params1 <- if (sum(labels == 1) < 5) {
    as.list(setNames(stable_fit_init(data), c("alpha", "beta", "gamma", "delta")))
  } else {
    estimate_stable_from_cdf(data[labels == 1], u)
  }

  params2 <- if (sum(labels == 2) < 5) {
    as.list(setNames(stable_fit_init(data), c("alpha", "beta", "gamma", "delta")))
  } else {
    estimate_stable_from_cdf(data[labels == 2], u)
  }

  log_likelihood_old <- -Inf

  for (iteration in 1:max_iter) {
    pdf1 <- pmax(r_stable_pdf(data, params1$alpha, params1$beta, params1$gamma, params1$delta), 1e-300)
    pdf2 <- pmax(r_stable_pdf(data, params2$alpha, params2$beta, params2$gamma, params2$delta), 1e-300)

    resp1 <- w * pdf1
    resp2 <- (1 - w) * pdf2
    total <- resp1 + resp2

    gamma1 <- resp1 / total
    gamma2 <- resp2 / total
    labels <- ifelse(gamma1 > gamma2, 1, 2)
    w <- mean(labels == 1)

    if (sum(labels == 1) >= 5) {
      best1 <- mock_gibbs_sampling(data[labels == 1])[[1]]
      params1 <- list(alpha = best1[2], beta = best1[3], gamma = best1[4], delta = best1[5])
    } else {
      message("Cluster 0 too small. Reusing previous estimate.\n")
    }

    if (sum(labels == 2) >= 5) {
      best2 <- mock_gibbs_sampling(data[labels == 2])[[1]]
      params2 <- list(alpha = best2[6], beta = best2[7], gamma = best2[8], delta = best2[9])
    } else {
      message("Cluster 1 too small. Reusing previous estimate.\n")
    }

    pdf1 <- pmax(r_stable_pdf(data, params1$alpha, params1$beta, params1$gamma, params1$delta), 1e-300)
    pdf2 <- pmax(r_stable_pdf(data, params2$alpha, params2$beta, params2$gamma, params2$delta), 1e-300)
    log_likelihood <- sum(log(w * pdf1 + (1 - w) * pdf2))

    message(sprintf("[Gibbs EM] Iteration %d: Log-Likelihood = %.6f\n", iteration, log_likelihood))

    if (abs(log_likelihood - log_likelihood_old) < tol) {
      message(sprintf("converged after %d iterations.\n", iteration))
      break
    }

    log_likelihood_old <- log_likelihood
  }

  return(list(params1 = params1, params2 = params2, weight = w))
}




