% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_observations.R
\name{get_observations}
\alias{get_observations}
\title{Observation Process}
\usage{
get_observations(
  sim_data,
  sim_results,
  type = c("random_one_layer", "random_all_layers", "from_data", "monitoring_based"),
  obs_error = c("rlnorm", "rbinom"),
  obs_error_param = NULL,
  ...
)
}
\arguments{
\item{sim_data}{\code{sim_data} object from \code{\link{initialise}} containing simulation
parameters}

\item{sim_results}{\code{sim_results} object; returned by \code{\link{sim}} function}

\item{type}{character vector of length 1; describes the sampling type
(case-sensitive):
\itemize{
\item "random_one_layer" - random selection of cells for which abundances
are sampled; the same set of selected cells is used across all time steps.
\item "random_all_layers" - random selection of cells for which abundances
are sampled; a new set of cells is selected for each time step.
\item "from_data" - user-defined selection of cells for which abundances
are sampled; the user is required to provide a \code{data.frame} containing
three columns: "x", "y" and "time_step".
\item "monitoring_based" - user-defined selection of cells for which
abundances are sampled; the user is required to provide a matrix object
with two columns: "x" and "y"; the abundance from given cell is sampled
by different virtual observers in different time steps; a geometric
distribution (\code{\link[stats:Geometric]{rgeom}}) is employed to define whether
a survey will be conducted by the same observer for several years or
not conducted at all.
}}

\item{obs_error}{character vector of length 1; type of the distribution
that defines the observation process: "\code{\link[stats:Lognormal]{rlnorm}}" (the log
normal distribution) or "\code{\link[stats:Binomial]{rbinom}}" (the binomial
distribution)}

\item{obs_error_param}{numeric vector of length 1; standard deviation
(on a log scale) of the random noise in observation process generated from
the log-normal distribution (\code{\link[stats:Lognormal]{rlnorm}}) or probability of
detection (success) when the binomial distribution
("\code{\link[stats:Binomial]{rbinom}}") is used.}

\item{...}{other necessary internal parameters:
\itemize{
\item \code{prop}

numeric vector of length 1; proportion of cells to be sampled
(default \code{prop = 0.1});
used when \verb{type = "random_one_layer" or "random_all_layers"},

\item \code{points}

\code{data.frame} or \code{matrix} with 3 numeric columns named "x", "y",
and "time_step" containing coordinates and time steps from which
observations should be obtained; used when \code{type = "from_data"},

\item \code{cells_coords}

\code{data.frame} or \code{matrix} with 2 columns named "x" and "y"; survey plots
coordinates; used when \code{type = "monitoring_based"}

\item \code{prob}

numeric vector of length 1; a parameter defining the shape of
\code{\link[stats:Geometric]{rgeom}} distribution; defines whether an observation
will be made by the same observer for several years, and whether it
will not be made at all (default \code{prob = 0.3});
used when \code{type = "monitoring_based"}

\item \code{progress_bar}

logical vector of length 1; determines if a progress bar for observation
process should be displayed (default \code{progress_bar = FALSE});
used when \code{type = "monitoring_based"}
}}
}
\value{
\code{data.frame} object with geographic coordinates, time steps,
estimated abundance, observation error (if \code{obs_error_param} is
provided), and observer identifiers (if \code{type = "monitoring_based"}). If \code{type = "from_data"}, returned object is sorted in the same order as the input \code{points}.
}
\description{
This function simulates an observation process. It accepts the \code{sim_results}
object, which is generated by the \code{\link{sim}} function, and applies the virtual
ecologist approach on the \code{N_map} component of the object. The function
returns a \code{data.frame} with the 'observed' abundances.
}
\examples{
\donttest{

library(terra)
n1_small <- rast(system.file("input_maps/n1_small.tif", package = "rangr"))
K_small <- rast(system.file("input_maps/K_small.tif", package = "rangr"))

# prepare data
sim_data <- initialise(
  n1_map = n1_small,
  K_map = K_small,
  r = log(2),
  rate = 1 / 1e3
)

sim_1 <- sim(obj = sim_data, time = 110, burn = 10)

# 1. random_one_layer
sample1 <- get_observations(
  sim_data,
  sim_1,
  type = "random_one_layer",
  prop = 0.1
)

# 2. random_all_layers
sample2 <- get_observations(
  sim_data,
  sim_1,
  type = "random_all_layers",
  prop = 0.15
)

# 3. from_data
sample3 <- get_observations(
  sim_data,
  sim_1,
  type = "from_data",
  points = observations_points
)

# 4. monitoring_based
# define observations sites
all_points <- xyFromCell(unwrap(sim_data$id), cells(unwrap(sim_data$K_map)))
sample_idx <- sample(1:nrow(all_points), size = 20)
sample_points <- all_points[sample_idx, ]

sample4 <- get_observations(
  sim_data,
  sim_1,
  type = "monitoring_based",
  cells_coords = sample_points,
  prob = 0.3,
  progress_bar = TRUE
)

# 5. noise "rlnorm"
sample5 <- get_observations(sim_data,
  sim_1,
  type = "random_one_layer",
  obs_error = "rlnorm",
  obs_error_param = log(1.2)
)

# 6. noise "rbinom"
sample6 <- get_observations(sim_data,
  sim_1,
  type = "random_one_layer",
  obs_error = "rbinom",
  obs_error_param = 0.8
)

}

}
