% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/osrm_start.R
\name{osrm_start}
\alias{osrm_start}
\title{Start an OSRM Server with Automatic Setup}
\usage{
osrm_start(
  path,
  algorithm = c("mld", "ch"),
  quiet = FALSE,
  verbose = FALSE,
  ...
)
}
\arguments{
\item{path}{A string. Path to the input data. Can be one of:
\itemize{
\item A path to an OSM file (e.g., \verb{/path/to/data.osm.pbf}).
\item A path to a directory containing OSRM graph files or an OSM file.
\item A direct path to a final graph file (\code{.osrm.mldgr} or \code{.osrm.hsgr}).
}}

\item{algorithm}{A string specifying the routing algorithm to use for graph
preparation, either \code{"mld"} (Multi-Level Dijkstra, default) or \code{"ch"}
(Contraction Hierarchies). This is only used when \code{osrm_prepare_graph} is
automatically called.}

\item{quiet}{A logical value. If \code{TRUE}, suppresses installer messages and
warnings. Defaults to \code{FALSE}.}

\item{verbose}{A logical. If \code{FALSE} (default), suppresses detailed console
output from backend commands. If \code{TRUE}, shows all output, which is useful
for debugging.}

\item{...}{Additional arguments passed on to \code{osrm_prepare_graph()} (e.g.,
\code{overwrite = TRUE}) and \code{osrm_start_server()} (e.g., \code{port = 5001}).}
}
\value{
A \code{processx::process} object for the running server.
}
\description{
A high-level, "one-shot" function to start an OSRM server that automatically
handles OSRM installation and graph preparation. This is the recommended
function for most users to get a server running quickly with minimal steps.
}
\details{
This function is designed for convenience and automates the entire setup
process. By default, it is not verbose and only prints high-level status
messages.

\enumerate{
\item \strong{Check for OSRM Installation:} It first verifies if the \code{osrm-routed}
executable is available in the system's \code{PATH}. If not, it automatically
calls \code{osrm_install()} to download and install the latest stable version.
\item \strong{Process Input Path and Prepare Graph:} The function intelligently
handles the \code{path} argument to find or create the necessary graph files.
If the graph files do not exist, it automatically runs \code{osrm_prepare_graph()}
to build them, which may take some time.
\item \strong{Start Server:} Once the graph files are located or prepared, it
launches the \code{osrm-routed} server and prints a confirmation message
with the server's PID and port.
}
For advanced users or for debugging, set \code{verbose = TRUE} to see the detailed
console output from the installation and graph preparation steps. For full
manual control, use the lower-level functions like \code{osrm_prepare_graph()} and
\code{osrm_start_server()} directly.
}
\examples{
\donttest{
if (identical(Sys.getenv("OSRM_EXAMPLES"), "true")) {
  install_dir <- osrm_install(
    version = "latest",
    path_action = "session",
    quiet = TRUE
  )

  # copy example OSM PBF into a temporary workspace to avoid polluting pkg data
  pbf_path <- system.file("extdata/cur.osm.pbf", package = "osrm.backend")
  osrm_dir <- file.path(tempdir(), paste0("osrm-", Sys.getpid()))
  dir.create(osrm_dir, recursive = TRUE)
  local_pbf <- file.path(osrm_dir, "cur.osm.pbf")
  file.copy(from = pbf_path, to = local_pbf, overwrite = TRUE)

  # Start the server with one command.
  # It will quietly install OSRM and prepare the graph if needed.
  osrm_process <- osrm_start(local_pbf)

  # Stop the server when done.
  osrm_stop()

  # To see all backend logs during setup, use verbose = TRUE
  osrm_process_verbose <- osrm_start(local_pbf, verbose = TRUE)
  osrm_stop()

  osrm_uninstall(
    dest_dir = install_dir,
    clear_path = TRUE,
    force = TRUE,
    quiet = TRUE
  )
  unlink(osrm_dir, recursive = TRUE)
}
}
}
\seealso{
\code{\link[=osrm_stop]{osrm_stop()}}, \code{\link[=osrm_start_server]{osrm_start_server()}} for manual server startup.
}
