#' @title Export bootstrapped parameters of polynomial model
#' @description Utilitary function to export bootstrapped parameters, as part of STEP3 of the 3-step identification strategy (see \code{RSAmodel}).
#' Utilitary function to export bootstrapped parameters
#' @param RSAbootstrap_object A matrix output generated by lavaan: \code{bootstrapLavaan(RSA_object$models$name_final, FUN="coef")}
#' @return A table of mean values and 95% confidence intervals of polynomial parameters
#' 
#' @examples
#' ###### Export 95% CI of bootstrapped estimates of polynomial
#' #Estimate a model: FM26_PARALLELASYMWEAK (simulation data)
#' RSA_NSfit  <- RSAmodel(formula= engagement ~ needs*supplies,
#' data= sim_NSfit, model= c("FM26_PARALLELASYMWEAK"))
#' #Bootstrapped sampling with lavaan
#' RSA_NSfit_boot  <- lavaan::bootstrapLavaan(RSA_NSfit$models$FM26_PARALLELASYMWEAK,
#'  R= 10,FUN="coef")
#' #Export results in a table
#' RSA_NSfit_boot_exp <- exportRSA.bootstrap(RSA_NSfit_boot)
#' RSA_NSfit_boot_exp
#' @export



exportRSA.bootstrap <- function(RSAbootstrap_object){
	
BOOTS_raw <- data.frame(RSAbootstrap_object)
BOOTS_raw$u1 <- BOOTS_raw$b1+ BOOTS_raw$b2
BOOTS_raw$u2 <- BOOTS_raw$b3+ BOOTS_raw$b4+ BOOTS_raw$b5
BOOTS_raw$u3 <- BOOTS_raw$b6+ BOOTS_raw$b7+ BOOTS_raw$b8+ BOOTS_raw$b9
BOOTS_raw$v1 <- BOOTS_raw$b1- BOOTS_raw$b2
BOOTS_raw$v2 <- BOOTS_raw$b3- BOOTS_raw$b4+ BOOTS_raw$b5
BOOTS_raw$v3 <- BOOTS_raw$b6- BOOTS_raw$b7+BOOTS_raw$b8- BOOTS_raw$b9

###Get mean estimate
BOOTS_mean <- BOOTS_raw[1,]
for(i in 1:ncol(BOOTS_raw)){
BOOTS_mean_i <- mean(BOOTS_raw[,i],na.rm=T)
BOOTS_mean[i] <- formatC(as.numeric(BOOTS_mean_i),digits=3,format="f")
}
BOOTS_mean

###Get 95% CI for estimate
BOOTS_95CI <- BOOTS_raw[1,]
for(i in 1:ncol(BOOTS_raw)){
BOOTS_25CI_i <- quantile(BOOTS_raw[,i],probs=c(.025))[1]
BOOTS_25CI_i  <- formatC(as.numeric(BOOTS_25CI_i),digits=3,format="f")
BOOTS_975CI_i <- quantile(BOOTS_raw[,i],probs=c(.975))[1]
BOOTS_975CI_i  <- formatC(as.numeric(BOOTS_975CI_i),digits=3,format="f")
BOOTS_95CI[i] <- paste("(", BOOTS_25CI_i,",", BOOTS_975CI_i,")",sep="")
}
BOOTS_95CI

### mean+95CI
BOOTS_merge <- BOOTS_raw[1,]
for(i in 1:ncol(BOOTS_raw)){
BOOTS_merge[i] <- paste(BOOTS_mean[i],BOOTS_95CI[i],sep=" ")
}
BOOTS_out <- cbind(Nboots=nrow(BOOTS_raw), BOOTS_merge)
BOOTS_out
}
