% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot.R
\name{plot.netsim}
\alias{plot.netsim}
\title{Plot Data from a Stochastic Network Epidemic Model}
\usage{
\method{plot}{netsim}(
  x,
  type = "epi",
  y,
  popfrac = FALSE,
  sim.lines = FALSE,
  sims,
  sim.col,
  sim.lwd,
  sim.alpha,
  mean.line = TRUE,
  mean.smooth = TRUE,
  mean.col,
  mean.lwd = 2,
  mean.lty = 1,
  qnts = 0.5,
  qnts.col,
  qnts.alpha = 0.5,
  qnts.smooth = TRUE,
  legend,
  leg.cex = 0.8,
  axs = "r",
  grid = FALSE,
  add = FALSE,
  network = 1,
  at = 1,
  col.status = FALSE,
  shp.g2 = NULL,
  vertex.cex,
  stats,
  targ.line = TRUE,
  targ.col,
  targ.lwd = 2,
  targ.lty = 2,
  plots.joined,
  duration.imputed = TRUE,
  method = "l",
  ...
)
}
\arguments{
\item{x}{An \code{EpiModel} model object of class \code{netsim}.}

\item{type}{Type of plot: \code{"epi"} for epidemic model results,
\code{"network"} for a static network plot (\code{plot.network}),
or \code{"formation"}, \code{"duration"}, or \code{"dissolution"} for
network formation, duration, or dissolution statistics.}

\item{y}{Output compartments or flows from \code{netsim} object to plot.}

\item{popfrac}{If \code{TRUE}, plot prevalence of values rather than numbers
(see details).}

\item{sim.lines}{If \code{TRUE}, plot individual simulation lines. Default is
to plot lines for one-group models but not for two-group models.}

\item{sims}{If \code{type="epi"} or \code{"formation"}, a vector of
simulation numbers to plot. If \code{type="network"}, a single
simulation number for which to plot the network, or else \code{"min"}
to plot the simulation number with the lowest disease prevalence,
\code{"max"} for the simulation with the highest disease prevalence,
or \code{"mean"} for the simulation with the prevalence closest to the
mean across simulations at the specified time step.}

\item{sim.col}{Vector of any standard R color format for simulation lines.}

\item{sim.lwd}{Line width for simulation lines.}

\item{sim.alpha}{Transparency level for simulation lines, where
0 = transparent and 1 = opaque (see \code{adjustcolor} function).}

\item{mean.line}{If \code{TRUE}, plot mean of simulations across time.}

\item{mean.smooth}{If \code{TRUE}, use a loess smoother on the mean line.}

\item{mean.col}{Vector of any standard R color format for mean lines.}

\item{mean.lwd}{Line width for mean lines.}

\item{mean.lty}{Line type for mean lines.}

\item{qnts}{If numeric, plot polygon of simulation quantiles based on the
range implied by the argument (see details). If \code{FALSE}, suppress
polygon from plot.}

\item{qnts.col}{Vector of any standard R color format for polygons.}

\item{qnts.alpha}{Transparency level for quantile polygons, where 0 =
transparent and 1 = opaque (see \code{adjustcolor} function).}

\item{qnts.smooth}{If \code{TRUE}, use a loess smoother on quantile polygons.}

\item{legend}{If \code{TRUE}, plot default legend.}

\item{leg.cex}{Legend scale size.}

\item{axs}{Plot axis type (see \code{\link{par}} for details), with default
of \code{"r"}.}

\item{grid}{If \code{TRUE}, a grid is added to the background of plot
(see \code{\link{grid}} for details), with default of nx by ny.}

\item{add}{If \code{TRUE}, new plot window is not called and lines are added
to existing plot window.}

\item{network}{Network number, for simulations with multiple networks
representing the population.}

\item{at}{If \code{type="network"}, time step for network graph.}

\item{col.status}{If \code{TRUE} and \code{type="network"}, automatic disease
status colors (blue = susceptible, red = infected, green = recovered).}

\item{shp.g2}{If \code{type="network"} and \code{x} is for a two-group model,
shapes for the Group 2 vertices, with acceptable inputs of "triangle"
and "square". Group 1 vertices will remain circles.}

\item{vertex.cex}{Relative size of plotted vertices if \code{type="network"},
with implicit default of 1.}

\item{stats}{If \code{type="formation","duration","dissolution"}, statistics
to plot.  For \code{type = "formation"}, \code{stats} are among those
specified in \code{nwstats.formula} of \code{\link{control.net}}; for
\code{type = "duration","dissolution"}, \code{stats} are among those
of the dissolution model (without \code{offset()}).  The default is
to plot all statistics.}

\item{targ.line}{If \code{TRUE}, plot target or expected value line for
the statistic of interest.}

\item{targ.col}{Vector of standard R colors for target statistic lines, with
default colors based on \code{RColorBrewer} color palettes.}

\item{targ.lwd}{Line width for the line showing the target statistic values.}

\item{targ.lty}{Line type for the line showing the target statistic values.}

\item{plots.joined}{If \code{TRUE} and
\code{type="formation","duration","dissolution"}, combine all
statistics in one plot, versus one plot per statistic if
\code{FALSE}.}

\item{duration.imputed}{If \code{type="duration"}, a logical indicating
whether or not to impute starting times for relationships extant at
the start of the simulation. Defaults to \code{TRUE} when
\code{type="duration"}.}

\item{method}{Plot method for
\code{type="formation","duration","dissolution"}, with options of
\code{"l"} for line plots and \code{"b"} for boxplots.}

\item{...}{Additional arguments to pass.}
}
\description{
Plots epidemiological and network data from a stochastic network
             model simulated with \code{\link{netsim}}.
}
\details{
This plot function can produce three types of plots with a stochastic network
model simulated through \code{\link{netsim}}:
\enumerate{
 \item \strong{\code{type="epi"}}: epidemic model results (e.g., disease
       prevalence and incidence) may be plotted.
 \item \strong{\code{type="network"}}: a static network plot will be
       generated. A static network plot of a dynamic network is a
       cross-sectional extraction of that dynamic network at a specific
       time point. This plotting function wraps the
       \code{\link{plot.network}} function in the \code{network} package.
       Consult the help page for \code{plot.network} for all of the plotting
       parameters. In addition, four plotting parameters specific to
       \code{netsim} plots are available: \code{sim}, \code{at},
       \code{col.status}, and \code{shp.g2}.
 \item \strong{\code{type="formation"}}: summary network statistics related
       to the network model formation are plotted. These plots are similar
       to the formation plots for \code{netdx} objects. When running a
       \code{netsim} simulation, one must specify there that
       \code{save.nwstats=TRUE}; the plot here will then show the network
       statistics requested explicitly in \code{nwstats.formula}, or will use
       the formation formula set in \code{netest} otherwise.
}

When \code{type="epi"}, this plotting function will extract the
epidemiological output from a model object of class \code{netsim} and plot
the time series data of disease prevalence and other results. The summary
statistics that the function calculates and plots are individual simulation
lines, means of the individual simulation lines, and quantiles of those
individual simulation lines. The mean line, toggled on with
\code{mean.line=TRUE}, is calculated as the row mean across simulations at
each time step.

Compartment prevalences are the size of a compartment over some denominator.
To plot the raw numbers from any compartment, use \code{popfrac=FALSE}; this
is the default for any plots of flows. The \code{popfrac} parameter
calculates and plots the denominators of all specified compartments using
these rules: 1) for one-group models, the prevalence of any compartment is
the compartment size divided by the total population size; 2) for two-group
models, the prevalence of any compartment is the compartment size divided by
the group population size. For any prevalences that are not automatically
calculated, the \code{\link{mutate_epi}} function may be used to add new
variables to the \code{netsim} object to plot or analyze.

The quantiles show the range of outcome values within a certain specified
quantile range. By default, the interquartile range is shown: that is the
middle 50\% of the data. This is specified by \code{qnts=0.5}. To show the
middle 95\% of the data, specify \code{qnts=0.95}. To toggle off the polygons
where they are plotted by default, specify \code{qnts=FALSE}.

When \code{type="network"}, this function will plot cross sections of the
simulated networks at specified time steps. Because it is only possible to
plot one time step from one simulation at a time, it is necessary to enter
these in the \code{at} and \code{sims} parameters. To aid in visualizing
representative and extreme simulations at specific time steps, the
\code{sims} parameter may be set to \code{"mean"} to plot the simulation in
which the disease prevalence is closest to the average across all
simulations, \code{"min"} to plot the simulation in which the prevalence is
lowest, and \code{"max"} to plot the simulation in which the prevalence is
highest.
}
\examples{
## SI Model without Network Feedback
# Initialize network and set network model parameters
nw <- network_initialize(n = 100)
nw <- set_vertex_attribute(nw, "group", rep(1:2, each = 50))
formation <- ~edges
target.stats <- 50
coef.diss <- dissolution_coefs(dissolution = ~offset(edges), duration = 20)

# Estimate the network model
est <- netest(nw, formation, target.stats, coef.diss, verbose = FALSE)

# Simulate the epidemic model
param <- param.net(inf.prob = 0.3, inf.prob.g2 = 0.15)
init <- init.net(i.num = 10, i.num.g2 = 10)
control <- control.net(type = "SI", nsteps = 20, nsims = 3,
                       verbose = FALSE, save.nwstats = TRUE,
                       nwstats.formula = ~edges + meandeg + concurrent)
mod <- netsim(est, param, init, control)

# Plot epidemic trajectory
plot(mod)
plot(mod, type = "epi", grid = TRUE)
plot(mod, type = "epi", popfrac = TRUE)
plot(mod, type = "epi", y = "si.flow", qnts = 1, ylim = c(0, 4))

# Plot static networks
par(mar = c(0, 0, 0, 0))
plot(mod, type = "network", vertex.cex = 1.5)

# Automatic coloring of infected nodes as red
par(mfrow = c(1, 2), mar = c(0, 0, 2, 0))
plot(mod, type = "network", main = "Min Prev | Time 50",
     col.status = TRUE, at = 20, sims = "min", vertex.cex = 1.25)
plot(mod, type = "network", main = "Max Prev | Time 50",
     col.status = TRUE, at = 20, sims = "max", vertex.cex = 1.25)

# Automatic shape by group number (circle = group 1)
par(mar = c(0, 0, 0, 0))
plot(mod, type = "network", at = 20, col.status = TRUE,
     shp.g2 = "square")
plot(mod, type = "network", at = 20, col.status = TRUE,
     shp.g2 = "triangle", vertex.cex = 2)

# Plot formation statistics
par(mfrow = c(1,1), mar = c(3,3,1,1), mgp = c(2,1,0))
plot(mod, type = "formation", grid = TRUE)
plot(mod, type = "formation", plots.joined = FALSE)
plot(mod, type = "formation", sims = 2:3)
plot(mod, type = "formation", plots.joined = FALSE,
     stats = c("edges", "concurrent"))
plot(mod, type = "formation", stats = "meandeg",
     mean.lwd = 1, qnts.col = "seagreen", mean.col = "black")

}
\seealso{
\code{\link{plot.network}}, \code{\link{mutate_epi}}
}
\keyword{plot}
