% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bpcRegTab.R
\name{bpcRegTab}
\alias{bpcRegTab}
\title{Classical and robust regression based on backwards pivot coordinates}
\usage{
bpcRegTab(
  X,
  y,
  obs.ID = NULL,
  row.factor = NULL,
  col.factor = NULL,
  value = NULL,
  external = NULL,
  norm.cat.row = NULL,
  norm.cat.col = NULL,
  robust = FALSE,
  base = exp(1),
  norm.const = F,
  seed = 8
)
}
\arguments{
\item{X}{object of class data.frame with columns corresponding to row and column factors of the respective compositional table, a variable with the values of the composition (positive values only) and a factor with observation IDs. The response y and non-compositional predictors can be also included.}

\item{y}{character with the name of response (if included in X), data frame with row names corresponding to observation IDs or a named array with values of the response.}

\item{obs.ID}{name of the factor variable distinguishing the observations. Needs to be given with the quotation marks.}

\item{row.factor}{name of the variable representing the row factor. Needs to be given with the quotation marks.}

\item{col.factor}{name of the variable representing the column factor. Needs to be given with the quotation marks.}

\item{value}{name of the variable representing the values of the composition. Needs to be given with the quotation marks.}

\item{external}{array with names of non-compositional predictors.}

\item{norm.cat.row}{the rationing category of the row factor. If not defined, all pairs are considered. Given in quotation marks.}

\item{norm.cat.col}{the rationing category of the column factor. If not defined, all pairs are considered. Given in quotation marks.}

\item{robust}{if TRUE, the MM-type estimator is used. Defaults to FALSE.}

\item{base}{a positive number: the base with respect to which logarithms are computed. Defaults to exp(1).}

\item{norm.const}{if TRUE, the regression coefficients corresponding to orthonormal coordinates are given a s result. Defaults to FALSE, the normalising constant is omitted.}

\item{seed}{a single value.}
}
\value{
A list containing:\describe{
\item{Summary}{the summary object which collects results from all coordinate systems. The names of the coefficients indicate the type of the respective coordinate 
(rbpb.1 - the first row backwards pivot balance, cbpb.1 - the first column backwards pivot balance and tbpc.1.1 - the first table backwards pivot coordinate) and the logratio or log odds-ratio quantified thereby. 
E.g. cbpb.1_C2.to.C1 would therefore correspond to the logratio between column categories C1 and C2, schematically written log(C2/C1), and tbpc.1.1_R2.to.R1.&.C2.to.C1 would correspond to the log odds-ratio computed from a 2x2 table, 
which is formed by row categories R1 and R2 and columns C1 and C2. See Nesrstova et al. (2023) for details.} 
\item{Base}{the base with respect to which logarithms are computed} 
\item{Norm.const}{the values of normalising constants (when results for orthonormal coordinates are reported).}
\item{Robust}{TRUE if the MM estimator was applied.}
\item{lm}{the lm object resulting from the first iteration.}
\item{Row.levels}{the order of the row factor levels cosidered in the first iteration.}
\item{Col.levels}{the order of the column factor levels cosidered in the first iteration.}
}
}
\description{
Performs classical or robust regression analysis of real response on a compositional table, which is represented in backwards pivot coordinates.  Also non-compositional covariates can be included (additively).
}
\details{
bpcRegTab

The set of compositional tables is repeatedly expressed in a set of backwards logratio coordinates, 
when each set highlights different combination of pairs of row and column factor categories, as detailed in Nesrstova et al. (2023).
For each coordinates system (supplemented by non-compositonal predictors), robust MM or classical least squares estimate of regression coefficients is performed and information respective to the first row, column and table backwards pivot coordinate is stored. 
The summary therefore collects results from several regression models, each leading to the same overall model characteristics, like the F statistics or R^2. 
In order to maintain consistency of the iterative results collected in the output, a seed is set before robust estimation of each of the models considered. Its specific value can be set via parameter seed.
}
\examples{
# let's prepare some data
data(employment2)
data(unemployed)

table_data <- employment2[employment2$Contract == "FT", ]
y <- unemployed[unemployed$age == "20_24" & unemployed$year == 2015,]
countries <- intersect(levels(droplevels(y$country)), levels(table_data$Country))

table_data <- table_data[table_data$Country \%in\% countries, ]
y <- y[y$country \%in\% countries, c("country", "value")]
colnames(y) <- c("Country", "unemployed")

# response as part of X
table_data.y <- merge(table_data, y, by = "Country")
reg.cla <- bpcRegTab(table_data.y, y = "unemployed", obs.ID = "Country", 
row.factor = "Sex", col.factor = "Age", value = "Value")
reg.cla

# response as named array
resp <- y$unemployed
names(resp) <- y$Country
reg.cla2 <- bpcRegTab(table_data.y, y = resp, obs.ID = "Country", 
row.factor = "Sex", col.factor = "Age", value = "Value")
reg.cla2

# response as data.frame, robust estimator, 55plus as the rationing category, logarithm of base 2
resp.df <- as.data.frame(y$unemployed)
rownames(resp.df) <- y$Country
reg.rob <- bpcRegTab(table_data.y, y = resp.df, obs.ID = "Country", 
row.factor = "Sex", col.factor = "Age", value = "Value",
norm.cat.col = "55plus", robust = TRUE, base = 2)
reg.rob

# Illustrative example with non-compositional predictors and response as part of X
x.ext <- unemployed[unemployed$age == "15_19" & unemployed$year == 2015,]
x.ext <- x.ext[x.ext$country \%in\% countries, c("country", "value")]
colnames(x.ext) <- c("Country", "15_19")

table_data.y.ext <- merge(table_data.y, x.ext, by = "Country")
reg.cla.ext <- bpcRegTab(table_data.y.ext, y = "unemployed", obs.ID = "Country", 
row.factor = "Sex", col.factor = "Age", value = "Value", external = "15_19")
reg.cla.ext
}
\references{
Nesrstova, V., Jaskova, P., Pavlu, I., Hron, K., Palarea-Albaladejo, J., Gaba, A., Pelclova, J., Facevicova, K. (2023). Simple enough, but not simpler: Reconsidering additive logratio coordinates in compositional analysis. Submitted
}
\seealso{
\code{\link{bpcTabWrapper}} 
\code{\link{bpcPcaTab}}
\code{\link{bpcReg}}
}
\author{
Kamila Facevicova
}
\keyword{regression}
