#' @title
#' Checking for partial ordering.
#'
#' @description
#' Checks whether the input binary relation is a partial order.
#'
#' @param set A list of character strings (the names of the elements of the set, on which the binary relation is defined).
#'
#' @param rel A two-columns character matrix, each row comprising an element (pair) of the binary relation.
#'
#' @return
#' A boolean value.
#'
#' @examples
#' set<-c("a", "b", "c", "d")
#'
#' rel <- matrix(c(
#'   "a", "b",
#'   "c", "b",
#'   "d", "a",
#'   "c", "a",
#'   "d", "b",
#'   "a", "a",
#'   "b", "b",
#'   "c", "c",
#'   "d", "d"
#' ), ncol = 2, byrow = TRUE)
#'
#' chk <- IsPartialOrder(set, rel)
#'
#' @name IsPartialOrder
#' @export IsPartialOrder
IsPartialOrder <- function(set, rel) {
  if (!is.character(set))
    stop("set must be character array")
  if (length(unique(set)) != length(set))
    stop("set contains duplicated values")
  if (ncol(rel) != 2)
    stop("rel must be two columns character matrix")
  if ((ncol(rel) * nrow(rel) > 0  && !is.character(rel)))
    stop("rel must be two columns character matrix")
  if (!(all(unique(as.vector(rel)) %in% set)))
    stop("rel contains values not belonging to set")

  tryCatch({
    result <- .Call("_isPartialOrder", set, rel)
    return (result)
  }, error = function(err) {
    err_split <- strsplit(err[[1]], split = ":")
    stop(err_split[[1]][length(err_split[[1]])])
  }) # END tryCatch
}
