% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ssurgo2sp.R
\name{ssurgo2sp}
\alias{ssurgo2sp}
\title{Take in \acronym{SSURGO} csv files and create a soil profile}
\usage{
ssurgo2sp(
  mapunit = NULL,
  component = NULL,
  chorizon = NULL,
  mapunit.shp = NULL,
  nmapunit = 1,
  nsoil = 1,
  xout = NULL,
  soil.bottom = 200,
  method = c("constant", "linear"),
  nlayers = 10,
  verbose = FALSE
)
}
\arguments{
\item{mapunit}{mapunit \acronym{SSURGO} file}

\item{component}{component \acronym{SSURGO} file}

\item{chorizon}{chorizon \acronym{SSURGO} file}

\item{mapunit.shp}{mapunit shapefile for creating metadata}

\item{nmapunit}{number of mapunits to select}

\item{nsoil}{number of soil components (within a mapunit) to consider}

\item{xout}{vector for interpolation and extrapolation}

\item{soil.bottom}{bottom of the soil profile}

\item{method}{method used for interpolation (see \code{\link{approx}})}

\item{nlayers}{number of soil layers to generate}

\item{verbose}{whether to print details of the process}
}
\value{
a list with soil profile matrices with length equal to nsoil
}
\description{
Utility function to convert \acronym{SSURGO} data to soil profile
}
\details{
Download the data from \acronym{SSURGO} using the \sQuote{FedData} package \cr
This will generate csv files \sQuote{chorizon}, \sQuote{component} and \sQuote{mapunit}, \cr
but also many other files which are not needed for creating a soil profile.
}
\examples{
\donttest{
require(ggplot2)
require(sf)
extd.dir <- system.file("extdata", package = "apsimx")

chorizon <- read.csv(paste0(extd.dir,"/ISUAG/SSURGO/ISUAG_SSURGO_chorizon.csv"))
component <- read.csv(paste0(extd.dir,"/ISUAG/SSURGO/ISUAG_SSURGO_component.csv"))
mapunit <- read.csv(paste0(extd.dir,"/ISUAG/SSURGO/ISUAG_SSURGO_mapunit.csv"))
mapunit.shp <- st_read(paste0(extd.dir,"/ISUAG/SSURGO/ISUAG_SSURGO_Mapunits.shp"), quiet = TRUE)

## Using default 'constant' method
sp.c <- ssurgo2sp(mapunit = mapunit, 
                 component = component, 
                 chorizon = chorizon, 
                 mapunit.shp = mapunit.shp)

sp.c <- sp.c[[1]]

ggplot(data = sp.c, aes(y = -Depth, x = Carbon)) + 
geom_point() + 
  geom_path() + 
  ylab("Soil Depth (cm)") + xlab("Organic Matter (percent)") +
  ggtitle("method = constant")
  
  
## Using 'linear' method
sp.l <- ssurgo2sp(mapunit = mapunit, 
                 component = component, 
                 chorizon = chorizon, 
                 mapunit.shp = mapunit.shp,
                 method = "linear")
                 
sp.l <- sp.l[[1]]
                 
ggplot(data = sp.l, aes(y = -Depth, x = Carbon)) + 
geom_point() + 
  geom_path() + 
  ylab("Soil Depth (cm)") + xlab("Organic Matter (percent)") +
  ggtitle("Method linear")
}

\dontrun{
## Method using get_ssurgo_tables

require(soilDB)
require(sp)
require(sf)
require(spData)
## retrieve data from lon -93, lat = 42
stbls <- get_ssurgo_tables(lonlat = c(-93, 42))

sp2.c <- ssurgo2sp(mapunit = stbls$mapunit, 
                 component = stbls$component, 
                 chorizon = stbls$chorizon, 
                 mapunit.shp = stbls$mapunit.shp)  
names(sp2.c)

metadata <- attributes(sp2.c[[1]])
metadata$names <- NULL; metadata$class <- NULL; metadata$row.names <- NULL

## Convert to an APSIM soil profile
asp2.c <- apsimx_soil_profile(nlayers = 10,
                              Thickness = sp2.c[[1]]$Thickness,
                              BD = sp2.c[[1]]$BD,
                              AirDry = sp2.c[[1]]$AirDry,
                              LL15 = sp2.c[[1]]$LL15,
                              DUL = sp2.c[[1]]$DUL,
                              SAT = sp2.c[[1]]$SAT,
                              KS = sp2.c[[1]]$KS,
                              Carbon = sp2.c[[1]]$Carbon,
                              PH = sp2.c[[1]]$PH,
                              metadata = metadata)
                              
plot(asp2.c)
plot(asp2.c, property = "water")

}
}
