% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/04_RF_CLUSTERING.R
\name{rf.clustering}
\alias{rf.clustering}
\title{Risk factor clustering}
\usage{
rf.clustering(db, metric, k = NA)
}
\arguments{
\item{db}{Data frame of risk factors supplied for clustering analysis.}

\item{metric}{Correlation metric used for distance calculation. Available options are:
\itemize{
\item \code{"raw pearson"} - calculated distance \code{as.dist(1 - cor(db, method = "pearson"))};
\item \code{"raw spearman"} - calculated distance \code{as.dist(1 - cor(db, method = "spearman"))};
\item \code{"common pearson"} - calculated distance \code{as.dist((1 - cor(db, method = "pearson")) / 2)};
\item \code{"common spearman"} - calculated distance \code{as.dist((1 - cor(db, method = "spearman")) / 2)};
\item \code{"absolute pearson"} - calculated distance \code{as.dist(1 - abs(cor(db, method = "pearson")))};
\item \code{"absolute spearman"} - calculated distance \code{as.dist(1 - abs(cor(db, method = "spearman")))};
\item \code{"sqrt pearson"} - calculated distance \code{as.dist(sqrt(1 - cor(db, method = "pearson")))};
\item \code{"sqrt spearman"} - calculated distance \code{as.dist(sqrt(1 - cor(db, method = "spearman")))};
\item \code{"x2y"} - calculated distance \code{as.dist(1 - dx2y(d = db)[[2]]))}.
}
\code{x2y} metric is proposed by Professor Rama Ramakrishnan and details can be found on this
\href{https://rama100.github.io/lecture-notes/x2y.nb.html}{link}. This metric is especially handy if
analyst wants to perform clustering before any binning procedures and to decrease number of risk factors. Additionally,
\code{x2y} algorithm process numerical and categorical risk factors at once and it is able to identify
non-linear relationship between the pairs. Metric \code{x2y} is not symmetric with respect to inputs - \code{x, y},
therefore arithmetic average of values between \code{xy} and \code{yx} is used to produce the final value for each pair.}

\item{k}{Number of clusters. If default value (\code{NA}) is passed, then automatic elbow method
will be used to determine the optimal number of clusters, otherwise selected number of clusters will be used.}
}
\value{
The function \code{rf.clustering} returns a data frame with: risk factors, clusters assigned and
distance to centroid (ordered from smallest to largest).
The last column (distance to centroid) can be used for selection of one or more risk factors per
cluster.
}
\description{
\code{rf.clustering} implements correlation based clustering of risk factors.
Clustering procedure is base on \link[stats]{hclust} from \code{stats} package.
}
\examples{
suppressMessages(library(PDtoolkit))
library(rpart)
data(loans)
#risk factors clustering using x2y metric
cr <- rf.clustering(db = loans[, -which(names(loans)\%in\%"Creditability")], 
		  metric = "x2y", 
		  k = 15)
cr
cr \%>\% group_by(clusters) \%>\% 
 slice(which.min(dist.to.centroid))
#clustering using common spearman metric
#first we need to categorize numeric risk factors
num.rf <- sapply(loans, is.numeric)
num.rf <- names(num.rf)[!names(num.rf)\%in\%"Creditability" & num.rf]
loans[, num.rf] <- sapply(num.rf, function(x) 
			  sts.bin(x = loans[, x], y = loans[, "Creditability"])[[2]])
#replace woe in order to convert to all numeric factors 
loans.woe <- replace.woe(db = loans, target = "Creditability")[[1]]
cr <- rf.clustering(db = loans.woe[, -which(names(loans.woe)\%in\%"Creditability")], 
		  metric = "common spearman", 
		  k = NA)
cr
#select one risk factor per cluster with min distance to centorid
cr \%>\% group_by(clusters) \%>\% 
 slice(which.min(dist.to.centroid))
}
