% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/maxlogL.R
\name{maxlogL}
\alias{maxlogL}
\title{Maximum Likelihood Estimation for parametric distributions}
\usage{
maxlogL(
  x,
  dist = "dnorm",
  fixed = NULL,
  link = NULL,
  start = NULL,
  lower = NULL,
  upper = NULL,
  optimizer = "nlminb",
  control = NULL,
  silent = FALSE,
  ...
)
}
\arguments{
\item{x}{a vector with data to be fitted. This argument must be a matrix
with hierarchical distributions.}

\item{dist}{a length-one character vector with the name of density/mass function
of interest. The default value is \code{'dnorm'}, to compute maximum likelihood
estimators of normal distribution.}

\item{fixed}{a list with fixed/known parameters of distribution of interest.
Fixed parameters must be passed with its name.}

\item{link}{a list with names of parameters to be linked, and names of the link function
object. For names of parameters, please visit documentation of density/mass
function. There are three link functions available: \code{\link{log_link}},
\code{\link{logit_link}} and \code{\link{NegInv_link}}.}

\item{start}{a numeric vector with initial values for the parameters to be estimated.}

\item{lower}{a numeric vector with lower bounds, with the same length of argument
\code{start} (for box-constrained optimization).}

\item{upper}{a numeric vector with upper bounds, with the same length of argument
\code{start} (for box-constrained optimization).}

\item{optimizer}{a lenght-one character vector with the name of optimization routine.
\code{\link{nlminb}}, \code{\link{optim}} and
\code{\link[DEoptim]{DEoptim}} are available; \code{\link{nlminb}}
is the default routine.}

\item{control}{control parameters of the optimization routine. Please, visit documentation of selected
optimizer for further information.}

\item{silent}{logical. If TRUE, warnings of \code{maxlogL} are suppressed.}

\item{...}{further arguments to be supplied to the optimizer.}
}
\value{
A list with class \code{"maxlogL"} containing the following lists:
\item{fit}{A list with output information about estimation.}
\item{inputs}{A list with all input arguments.}
\item{outputs}{A list with some output additional information:
\itemize{
\item Number of parameters.
\item Sample size
\item Standard error computation method.
}
}
}
\description{
Function to compute maximum likelihood estimators (MLE)
of any distribution implemented in \code{R}.
}
\details{
\code{maxlogL} computes the likelihood function corresponding to
the distribution specified in argument \code{dist} and maximizes it through
\code{\link{optim}}, \code{\link{nlminb}} or \code{\link{DEoptim}}. \code{maxlogL}
generates an S3 object of class \code{maxlogL}.

Noncentrality parameters must be named as \code{ncp} in the distribution.
}
\note{
The following generic functions can be used with a \code{maxlogL} object:
\code{summary, print, AIC, BIC, logLik}.
}
\examples{
library(EstimationTools)

#--------------------------------------------------------------------------------
# Example 1: estimation with one fixed parameter
x <- rnorm(n = 10000, mean = 160, sd = 6)
theta_1 <- maxlogL(x = x, dist = 'dnorm', control = list(trace = 1),
                 link = list(over = "sd", fun = "log_link"),
                 fixed = list(mean = 160))
summary(theta_1)


#--------------------------------------------------------------------------------
# Example 2: both parameters of normal distribution mapped with logarithmic
# function
theta_2 <- maxlogL(x = x, dist = "dnorm",
                   link = list(over = c("mean","sd"),
                               fun = c("log_link","log_link")))
summary(theta_2)


#--------------------------------------------------------------------------------
# Example 3: parameter estimation in ZIP distribution
if (!require('gamlss.dist')) install.packages('gamlss.dist')
library(gamlss.dist)
z <- rZIP(n=1000, mu=6, sigma=0.08)
theta_3  <- maxlogL(x = z, dist='dZIP', start = c(0, 0), lower = c(-Inf, -Inf),
                   upper = c(Inf, Inf), optimizer = 'optim',
                   link = list(over=c("mu", "sigma"),
                   fun = c("log_link", "logit_link")))
summary(theta_3)


#--------------------------------------------------------------------------------
# Example 4: parameter estimation with fixed noncentrality parameter.
y_2 <- rbeta(n = 1000, shape1 = 2, shape2 = 3)
theta_41 <- maxlogL(x = y_2, dist = "dbeta",
                    link = list(over = c("shape1", "shape2"),
                    fun = c("log_link","log_link")))
summary(theta_41)

# It is also possible define 'ncp' as fixed parameter
theta_42 <- maxlogL(x = y_2, dist = "dbeta", fixed = list(ncp = 0),
                    link = list(over = c("shape1", "shape2"),
                    fun = c("log_link","log_link")) )
summary(theta_42)


#--------------------------------------------------------------------------------

}
\references{
\insertRef{Nelder1965}{EstimationTools}

\insertRef{Fox1978}{EstimationTools}

\insertRef{Nash1979}{EstimationTools}

\insertRef{Dennis1981}{EstimationTools}
}
\seealso{
\code{\link{summary.maxlogL}}, \code{\link{optim}}, \code{\link{nlminb}},
\code{\link{DEoptim}}, \code{\link{DEoptim.control}},
\code{\link{maxlogLreg}}, \code{\link{bootstrap_maxlogL}}

Other maxlogL: 
\code{\link{maxlogLreg}()}
}
\author{
Jaime Mosquera Gutiérrez, \email{jmosquerag@unal.edu.co}
}
\concept{maxlogL}
